% Script to calculate and display various diagnostics from the 2D non-hydrostatic channel
% model runs. Reads in data files and then displays plots from the following options
% theta & angular momentum , flow, Ertel's Boussinesq PV, tracer, Richardson No., 
% non-hydrostatic parameter and bird's eye views of vertical vorticity and temperature.
%
% Author Tom, 13/7/95. 

clear
clear global
close all
whos

global x_spacing y_spacing z_spacing NX NY NZ
global u_x_grid v_x_grid w_x_grid p_x_grid PV_x_grid RiNo_x_grid xvort_x_grid yvort_x_grid zvort_x_grid
global u_y_grid v_y_grid w_y_grid p_y_grid PV_y_grid RiNo_y_grid xvort_y_grid yvort_y_grid zvort_y_grid
global u_z_grid v_z_grid w_z_grid p_z_grid PV_z_grid RiNo_z_grid xvort_z_grid yvort_z_grid zvort_z_grid
global x_domain y_domain z_domain
global u v w theta q E ang_mom trac RiNo delta zvort slice
global f g c N2 alpha rho_bar flux time this_path Half_flux
global zoom_x zoom_y cooling_width cooling_offset


% Plot cross section of this slice
slice = 1 ;

% Region to zoom in on. (x in km; y in m)
zoom_x = [0 30] ;
zoom_y = [-1000 0] ;

% Aspect Ratio.
AsRat = 0.075*1000*(zoom_x(2)-zoom_x(1))/(zoom_y(2)-zoom_y(1)) ;

% Constants. SI units unless explicitly stated otherwise.
Earth_rad  = 6370e3 ;
Radian     = 57.29578 ;
g          = 9.81 ;
c          = 4e3 ;
rho_bar    = 1000 ; 
True       = 1 ;
False      = 0 ;
secs_in_a_day = 60*60*24 ;

% Local variables.
getkey = 'y' ;

% Read in list of parameters for this session. 
param_file_name =   ...
   input(' Please enter the name of the m-file with the parameters for this run : ','s') ;
folder = input(' Please enter the number of the subdirectory for this experiment : ','s') ;

time    = str2num(folder) * 60 ;
info    = [param_file_name,' Day ',num2str(str2num(folder) * 60/secs_in_a_day),' (',folder,') '] ;
title1  = [info,' Temperature (oC) and angular momentum.' ];
message = [' Hit <y> to see a plot of ',title1] ;
getkey1 = input( message,'s') ;

title3  = [info,' Potential vorticity normalised by initial value.'] ;
message = [' Hit <y> to see a plot of ',title3] ;
getkey3 = input( message,'s') ;

title4  = [info,' Tracer, normalised by initial value.'] ;
message = [' Hit <y> to see a plot of ',title4] ;
getkey4 = input( message,'s') ;

fprintf(1,' \n\n') ;

feval(param_file_name) ;
this_path   = [this_path,'/',folder] ;

% Define grids.
setup_grids(x_spacing,y_spacing,z_spacing) ;

% Get surface fluxes.
flux     = Half_flux.*(tanh((2.*(p_y_grid - cooling_offset))./cooling_width) + 1 ) ;
flux     = [p_y_grid',flux'] ;

% Read data using external function readmodel.m. 
% Read temperature (units C).
theta = readmodel('theta.sun.b',this_path,'float32') ;

% Read u (units m/s). 
u = readmodel('u.sun.b',this_path,'float32') ;

% Read v (units m/s). 
v = readmodel('v.sun.b',this_path,'float32') ;

% Read w (units Nm^-2s^-1; divide by -(g*rho_bar) to get m/s, +ve upwards). 
w = readmodel('w.sun.b',this_path,'float32') ;
w = w./(-g*rho_bar) ;


% Print figures, of theta & ang_mom, flow, PV tracer and Ri/delta, with upper panel 
% showing a slice and lower panel the along channel mean.

if(getkey1 == 'y' | getkey1 == 'Y')

% Rough calculation of angular momentum.
   fprintf(1,' Calculating angular momentum ... \n') ;
   ang_mom = zeros(NY,NZ*NX) ;
   temp  = reshape(u,NY,NX*NZ) ;
   temp2 = meshgrid(f.*u_y_grid,[1:NX*NZ])' ;
   ang_mom = temp - temp2 ;
   ang_mom = reshape(ang_mom,NY*NZ,NX) ;
   clear temp ;
   clear temp2 ;

   theta_fig(title1) ;
   set(gca,'AspectRatio',[AsRat,NaN]) ;
   flow_fig(title1) ;
   %clear global ang_mom ;
   orient landscape

% Calculate net flow along channel (should be zero).
   new_u = regrid(u,'x',u_x_grid,p_x_grid,x_domain) ; 
   mask = (abs(new_u) < 10*eps) ;
   total = volume_integ(new_u,x_spacing,y_spacing,z_spacing,mask)/x_spacing ;
   if(total > 100*eps)
      fprintf(1,' ERROR !  There is net momentum flux along the channel !') ;
   end
end

if(getkey3 == 'y' | getkey3 == 'Y')

% Get PV. This function also calculates the vertical vorticity, the RiNo and delta
% (non-hydrostatic parameter) for free !  These are written to disk for later use.
   temp = [this_path,'/','q.mat'] ;
   if( fopen(temp) == -1)
      PV ;
      fid = fopen(temp,'w') ;
      fwrite(fid,q,'float32') ;
%  Reload model variables (PV function clears them).
      global theta u v w
      theta = readmodel('theta.sun.b',this_path,'float32') ;
      u = readmodel('u.sun.b',this_path,'float32') ;
      v = readmodel('v.sun.b',this_path,'float32') ;
      w = readmodel('w.sun.b',this_path,'float32') ;
      w = w./(-g*rho_bar) ;
   else
      fprintf(1,' Retrieving PV from disk ... \n') ;
      fid = fopen(temp,'r') ;
      q = fread(fid,[NY*NZ,NX],'float32') ;
   end
   fclose(fid) ;
   q = -q.*g./(f.*N2) ;
   clear temp

   PV_fig(title3) ;
   set(gca,'AspectRatio',[AsRat,NaN]) ;
   flow_fig(title3) ;
   orient landscape
end

if(getkey4 == 'y' | getkey4 == 'Y')
   trac = readmodel('salt.sun.b',this_path,'float32')./36 ;
   tracer_fig(title4) ;
   set(gca,'AspectRatio',[AsRat,NaN]) ;
   flow_fig(title4) ;
   clear trac  
   orient landscape
end


