% Script to read and display the mixed layer depth and density field from the CME model.
% Script can also read Levitus data files and output from a simple analytic model (Hnet3.m).
% Depth and density fields come from dumps of T and S, then processed by Mick's fortran 
% code. Code then calculates the flux in the mixed layer at each grid point due to the 
% baroclinic eddies. The divergence of this flux is calculated, and this represents the
% extra contribution to the vertical heat budget of the mixed layer. See notes in MIT 
% lab book, and in loose sheets.  
% 29/8/94. Add diagnostic to plot up K too.
% 1/9/94.  Find error in calculation of gradients from spherical coordinate system. Fix it.
% 13/9/94. CME model uses biharmonic diffusion. Calculate equivalent map of Laplacian
%          diffusion to compare directly with the mixed layer eddy K.
% 
% CONST_f is a variable set to True if diagnostic runs are being examined, and a constant
% value of f is required.
% DEBUG is set to True if the 2-D simplified form for the eddy flux is required (used for
% cases with no zonal variation).

% Author Tom July, 1994.

close
close
close
close

% Constants. SI units unless explicitly stated otherwise.
% rhobar and f are calculated explicitly.
Earth_rad  = 6370e3 ;
Radian     = 57.29578 ;
alpha      = 2e-4 ;
g          = 9.81 ;
gamma      = 0.3 ;
c          = 4e3 ;
Omega      = 7.292e-5 ;
True       = 1 ;
False      = 0 ;

fprintf(1,' Please enter the data type you want to work on ;\n') ;
data_type = 0 ;
while (data_type < 1 | data_type >6)
  data_type = input(' (1 = 1 deg CME, 2 = 1/3 deg CME, 3 = Levitus Atlantic, 4 = Levitus global, 5 = Levitus Pacific, 6 = Hnet3 output)  :  ')  ;
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Setup variables relevant to data format.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

global CONST_f DEBUG
global No_x_pts No_y_pts No_vars data_format data_path data_file_name
global dxdeg dydeg longstart latstart y_zoom x_zoom tropics
global ml_depth_cont_levels ml_rho_cont_levels flux_cont_levels heat_cont_levels ...
       K_cont_levels

if (data_type == 1)
  setupCME1deg
elseif (data_type == 2) 
  setupCME1_3deg
elseif (data_type == 3)
  setupLevitusAtl
elseif (data_type == 4)
  setupLevitusglob
elseif (data_type == 5)
  setupLevitusPac
elseif (data_type == 6)
  setupHnet3output
  flux_limit_lo = -200 ;
  flux_limit_hi = -0  ;
  Hnet3_f  = 1e-4 ;
end

data_file_name = input(' Please enter the data file name :  ','s') ;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Start doing calculations here.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Setup matrices to give the latitude and longitude of each point in the data matrices.
% These are constructed from the spacing, size, and start information using meshgrid fn.
x_space = [0:No_x_pts-1] ;
y_space = [0:No_y_pts-1] ;
[long,lat] = meshgrid((x_space.*dxdeg + longstart),(y_space.*dydeg + latstart)) ;

% Set longitudes into 0->360.
long = rem(long,360) ;

% Setup constant spacing variables in radians. Data format is assumed to be arranged on
% evenly spaced spherical coordinates.
dlong_space = dxdeg / Radian ; 
dlat_space  = dydeg / Radian ; 

% Caluclate f for each grid point. Set f to NaN for portion of data in the tropics.
f = 2 * Omega * sin(lat./Radian) ;
f(tropics,:) = NaN .* f(tropics,:) ;

% Read in data.
ml_depth = zeros(No_x_pts,No_y_pts*3)  ;
ml_rho   = zeros(No_x_pts,No_y_pts*3)  ;

if(~(exist('data')))
   fprintf(1,'Starting read ...') ;
   data = read([data_path,data_file_name], data_format,No_x_pts,No_y_pts,No_vars) ;
end

fprintf(1,'Calculating ...') ;

% Set up matrices for matlab to do contouring.
ml_depth = data([1:No_y_pts],:) ;
ml_rho   = data([No_y_pts + 1:2*No_y_pts],:) ;

ml_x     = ml_depth(:,[1:No_x_pts]) ;
ml_y     = ml_depth(:,[No_x_pts+1:2*No_x_pts]) ;
ml_depth = ml_depth(:,[2*No_x_pts+1:3*No_x_pts]) ;
ml_rho   = ml_rho(:,[2*No_x_pts+1:3*No_x_pts]) ;

% Convert density to sigma0 (units kg/m^3)
ml_rho   = ml_rho - 1000. ;

% Remove land values from matrices. Can't find a neat way to do this with matrix ops.
mask = find(ml_depth == -100) ;
temp = NaN.*ones(size(mask)) ;
ml_depth(mask) = temp ;
ml_rho(mask)   = temp ;
land = isnan(ml_depth) ;
clear temp mask

% Calculate grad(ml_rho) at each grid point.
% Wherever gradients are calculated here the are done so with respect to the lat and long
% in radians, then multiplied by the local value of d(lat)/dy and d(long)/dx. These are
% equal to 1/Earth_rad, and 1/(Earth_rad * cos(lat/Radian)) ;
[drhodx,drhody] = gradient(ml_rho,dlong_space,dlat_space) ;
drhodx = drhodx ./ (Earth_rad .* cos(lat/Radian)) ;
drhody = drhody ./ Earth_rad ;
mod_grad_rho  = sqrt(drhodx.^2 + drhody.^2) ;

% Calculate rhobar from the actual density field.
temp   = ml_rho ;
temp(isnan(temp)) = [] ;
rhobar = 1000. + mean(mean(temp)) ;

% Form the flux due to baroclinic eddies in the mixed layer (Tom's notes). 
% Note that flux here is in W/m^2, so must be multiplied by the mixed layer depth before the
% divergence is taken, and the equivalent vertical flux found.
if CONST_f
   const2    = ((gamma * c * g^2) / (alpha * Hnet3_f^3 * rhobar^2)) ;
   flux  = const2 * ml_depth.^2 .* mod_grad_rho.^2;
else
   const     = ((gamma * c * g^2) ./ (alpha .* f.^3 * rhobar^2)) ;
   flux  = const .* ml_depth.^2 .* mod_grad_rho.^2;
end
flux_x = flux .* drhodx ;
flux_y = flux .* drhody ;
mod_flux = sqrt(flux_x.^2 + flux_y.^2) ;

% Get max & min fluxes to print on plot.
temp = mod_flux(y_zoom,x_zoom) ;
temp(isnan(temp)) = [] ;
max_flux = max(temp) ;
min_flux = min(temp) ;

if DEBUG  
% Get flux from full expression.
    [temp,Hsurf_a] = gradient((flux_y.*ml_depth),dlong_space,dlat_space) ;
    Hsurf_a = Hsurf_a ./ Earth_rad ;

% Alternative form for flux from 1-D expressions (see MIT lab book).
    ml_depth2 = mean(ml_depth')' ;
    drhody2   = mean(drhody')' ; 
    flux2 = const2 .* ml_depth2.^2 .*  drhody2.^3 ;
    Hsurf_b = gradient((flux2.*ml_depth(:,1))',dlat_space) ;
    Hsurf_b = Hsurf_b ./ Earth_rad ;

% Plot up fluxes and heats from 3D and 2D expressions.
    y_grid = [1 : No_y_pts] ;
    figure
    subplot(2,2,1) ;
    plot(y_grid/1000.,flux2) ;
    xlabel('Distance (km)') ;
    ylabel('Flux from Hnet3 expression (W/m^-2)') ;
    
    subplot(2,2,2) ;
    plot(y_grid/1000.,flux_y(:,1)) ;
    xlabel('Distance (km)') ;
    ylabel('Flux from CME.m expression (W/m^-2)') ;
     
    subplot(2,2,3) ;
    plot(y_grid/1000.,Hsurf_b) ;
%   axis([0,y_grid(No_y_pts)/1000.,flux_limit_lo,flux_limit_hi]) ;
    axis([0,y_grid(No_y_pts)/1000.,0,max(max(Hsurf_b))]) ;
    xlabel('Distance (km)') ;
    ylabel('Vertical flux from Hnet3 expression (W/m^-2)') ;
    
    subplot(2,2,4) ;
    plot(y_grid/1000.,Hsurf_a) ;
%   axis([0,y_grid(No_y_pts)/1000.,flux_limit_lo,flux_limit_hi]) ;
    axis([0,y_grid(No_y_pts)/1000.,0,max(max(Hsurf_a))]) ;
    xlabel('Distance (km)') ;
    ylabel('Vertical flux from CME.m expression (W/m^-2)') ;
end

% Get effective heating due to eddies from the divergence of the flux.
[temp1, temp] = gradient((flux_x.*ml_depth),dlong_space,dlat_space) ;
temp1 = temp1 ./ (Earth_rad .* cos(lat/Radian)) ;
[temp, temp2] = gradient((flux_y.*ml_depth),dlong_space,dlat_space) ;
temp2 = temp2 ./ Earth_rad ;

heat  = temp1 + temp2 ;

% Calculate max, min and mean of the eddy divergence.
temp   = heat ;
temp(isnan(temp)) = [] ;
heatbar = mean(mean(temp)) ;
heatmax = max(max(temp)) ;
heatmin = min(min(temp)) ;

% Get K, the value of the effective diffusion coefficient.
if CONST_f
   K = (gamma * g^2 / rhobar^2 ) .* ml_depth.^2 .* mod_grad_rho.^2 / Hnet3_f^3 ;
else
   K = (gamma * g^2 / rhobar^2 ) .* ml_depth.^2 .* mod_grad_rho.^2 ./ f.^3 ;
end
max_K = max(K(~isnan(K))) ;
min_K = min(K(~isnan(K))) ;
mean_K = mean(K(~isnan(K))) ;

% If the data_type is CME 1/3 deg calculate the effective K from the model itself. Model
% has biharmonic diffusion with K' = 2.5e15 m^4s^-1.
if (data_type == 2)
   K_p = 2.5e15 ;
   equiv_K = K_p .* del2(del2(ml_rho))./del2(ml_rho) ;
   max_equiv_K = max(equiv_K(~isnan(equiv_K))) ;
   min_equiv_K = min(equiv_K(~isnan(equiv_K))) ;
   mean_equiv_K = mean(equiv_K(~isnan(equiv_K))) ;

end

fprintf(1,'   Finished.\n\n') ;

% Print out these stats.
fprintf(1,'Max, mean & min. heat convergence due to eddies : %g %g %g. \n', ...
       heatmax,heatbar,heatmin) ;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Do plots.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Do contour plots of mixed layer depth and density.
map(ml_x(y_zoom,x_zoom),ml_y(y_zoom,x_zoom),ml_depth(y_zoom,x_zoom),   ...
    land(y_zoom,x_zoom),ml_depth_cont_levels,'none',True, ...
    ['Mixed layer depth (m). ',data_file_name]);
getkey = input('Hit <Y> to add on down gradient arrows of eddy heat flux.    ','s') ;
if (getkey == 'Y' | getkey == 'y')
    hold on
    quiver(ml_x,ml_y,flux_x,flux_y,15) ;
    temp = ...
['Arrow plot of eddy heat flux (W/m^2). (Max,Min) = (',num2str(max_flux),',',num2str(min_flux),').'] ;
    text(x_zoom(1),y_zoom(1)-0.1*(max(y_zoom)-min(y_zoom)),temp) ;
    hold off
end

map(ml_x(y_zoom,x_zoom),ml_y(y_zoom,x_zoom),ml_rho(y_zoom,x_zoom),   ...
    land(y_zoom,x_zoom),ml_rho_cont_levels,'none',True, ...
    ['Mixed layer density (sigma0 : kg/m^3). ',data_file_name]);
getkey = input('Hit <Y> to add on down gradient arrows of eddy heat flux.    ','s') ;
if (getkey == 'Y' | getkey == 'y')
    hold on
    quiver(ml_x,ml_y,flux_x,flux_y,15) ;
    temp = ...
['Arrow plot of eddy heat flux (W/m^2). (Max,Min) = (',num2str(max_flux),',',num2str(min_flux),').'] ;
    text(x_zoom(1),y_zoom(1)-0.1*(max(y_zoom)-min(y_zoom)),temp) ;
    hold off
end


% Plot contour map of flux magnitude.
map(ml_x(y_zoom,x_zoom),ml_y(y_zoom,x_zoom),mod_flux(y_zoom,x_zoom),   ...
    land(y_zoom,x_zoom),flux_cont_levels,'none',True, ...
    ['Magnitude of lateral eddy heat flux (W/m^2). ',data_file_name]);

% Plot map of eddy heating.
map(ml_x(y_zoom,x_zoom),ml_y(y_zoom,x_zoom),heat(y_zoom,x_zoom),   ...
    land(y_zoom,x_zoom),heat_cont_levels,'none',True,  ...
    ['Eddy heat flux divergence (W/m^2). ',data_file_name]);

% Plot map of eddy heating.
map(ml_x(y_zoom,x_zoom),ml_y(y_zoom,x_zoom),K(y_zoom,x_zoom),   ...
    land(y_zoom,x_zoom),K_cont_levels,'none',True,  ...
    ['Eddy diffusion coefficient (m^2/s). ',data_file_name]);
hold on
temp = ['(Maximum, Minimum) = (',num2str(max_K),',',num2str(min_K),').'] ;
text(x_zoom(1),y_zoom(1)-0.1*(max(y_zoom)-min(y_zoom)),temp) ;
hold off

% For CME 1/3 deg model offer option of mapping the equivalent Laplacian diffusion
% coefficeint for the biharmonic diffusion actually used.
if(data_type == 2 )
  map(ml_x(y_zoom,x_zoom),ml_y(y_zoom,x_zoom),equiv_K(y_zoom,x_zoom),   ...
      land(y_zoom,x_zoom),K_cont_levels,'none',True,  ...
      ['CME 1/3 deg equivalent Laplacian diffusion coefficient (m^2/s). ',data_file_name]);
end

