% Script to calculate the net vertical heat required to balance the heat transport by
% baroclinic eddies in the mixed layer. Different version to Hnet.m which is an early
% attempt and does not relate the underlying N to the drhody & dHdy.
% This version takes a generalised quadratic mixed layer variation and calculates the
% shape of the surface heat flux which this corresponds to in a steady state.
% Requires input of the mixed layer shape, ambient N, and form for K (lateraal mixing
% coefficient.

% The eqn that Hsurf satisfies is :
% H d^2H/dy^2 + (dH/dy)^2 = g/(K rhobar N^2) Hsurf
% where K is CONSTANT.

% Author : Tom, May '94.

clear

% Setup model parameters, and constants. (SI units throughout).

g      = 10 ;
c      = 4e3 ;
rhobar = 1025 ;
alpha  = 1e-4 ;
f      = 1e-4 ;

Y_lo   = 0 ;
Y_hi   = 5000e3 ;
No_pts = 50 ;

% Setup model variables. Require input of H(y) which is quadratic in y.  Also require
% N (constant) and K.

Hzero      = 100  ;
Hmax       = 400  ;
Hmid       = 1.33*(Hmax + Hzero) / 2 ;

N          = 50*1e-4 ;

% Calculate parameters required for solution.

linear_H   = (4*Hmid - 3*Hzero - Hmax) / Y_hi      ;
quad_H     = (Hmax   - Hzero - linear_H*Y_hi) / Y_hi^2 ;

y_grid = [Y_lo : (Y_hi - Y_lo)/No_pts : Y_hi]  ;
H      = Hzero + linear_H.*y_grid + quad_H.*y_grid.*y_grid ;

d2Hdy2 = 2*quad_H ;
dHdy   = linear_H + 2*quad_H.*y_grid ;

% Assume constant background stratification with no sloping of isopycnals. Hence
% drhody = drhodz dHdy.
% Since drhodz = rhobar N^2 / g,  drhody = (rhobar N^2 / g ) * dHdy

drhody = (rhobar*N^2/g) * dHdy ;
rho    = rhobar + (rhobar*N^2/g) .* (H - Hzero) ;

% K here as per Martin's parameterisation, but considered constant for now.
% He puts <u`rho`> ~ (g H^2 N / (rhobar f^2)) (drhodx)^2
% So flux of heat goes as c/alpha * <u'rho'> 
% The value of K is thus ~ (c g N H^2 drhody) / (alpha rhobar  f^2)
% Note that the N here is the value IN THE MIXED LAYER; not the ambient value.
N_mix      = 0.5*f;
K          = (c * g * N_mix * mean(H.^2) * mean(drhody) ) / (rhobar * f^2 * alpha)


% Calculate Hnet1 as a function of y.

Hnet1  = (K * rhobar * N^2 / g) * (d2Hdy2.*H + dHdy.*dHdy) ;

subplot(3,1,1) ;
axis([Y_lo Y_hi -max(H) 0.0]) ;
plot(y_grid./1000,-H) ;
xlabel('Distance (km)') ;
ylabel('Mixed layer depth (m)') ;
text1 = ['K = ',num2str(K),', thermocline N/f = ',num2str(N/f)] ;
text(0,-50, text1)

subplot(3,1,2) ;
plot(y_grid./1000,rho-1000) ;
xlabel('Distance (km)') ;
ylabel('Mixed layer sigma (kgm^-3)') ;

subplot(3,1,3) ;
plot(y_grid./1000,Hnet1) ;
xlabel('Distance (km)') ;
ylabel('Hnet (Wm^-2)') ;


