% Script to calculate the vertical heat corresponding to the divergence of the heat from
% baroclininc eddies in the mixed layer. This one calculates the mixed layer profile
% by integrating the surface heat flux which balances the eddy flux divergence.
% The mixing parameterisation here is due to Martin, with the full formulation
% implemented. See hand written sheets for eqns which are solved.

% Require a separate function to do the heat flux, which is then numerically
% integrated using quad. NB Hsurf +ve means flux of heat INTO the sea.

% The eqn which this script solves is :
% H^2 = H0^2 + 2.const. integ(- ( integ( Hsurf dy ) - BCterm )^1/3 dy
% where the const = f/Nth^2 * (alpha g / (gamma c rhobar))^1/3
% and Nth is the ambient stratification of the motionless thermocline and gamma is the
% dimensionless constant for the eddy efficiency.

% The boundary conditions are chosen such that you specify the flux at the Northern
% wall = 0 (at y= Y_hi) and the depth at y=0. Since there is heat conservation the
% flux at y=0 is effectively given too. BCterm is a const equal to the integral of
% Hsurf from Y_lo to Y_hi, and used by the HsurfInteg.m function.

% 25/7/94. Add lines at the end of the script to dump the mixed layer fields to disk. This
% is so the CME.m script can be tested with these diagnostic fields.

% 27/7/94. Add in gamma term. Add code at end to compare different expressions about the
% mixed layer flux for use as diagnostics in  CME.m

% 29/7/94. Add in code to calculate the slope of the mixed layer base with respect to the
% maximum allowed (based on the radius of deformation). Also calculate the magnitude of an
% extra term which is omitted in the the derivation of these expressions. It comes in for
% strongly sloping mixed layer bases, and should be compared to unity (less than this means
% it has little effect).

% 1/8/94. Add in code with BC_type. This allows choice of different boundary conditions.
% BC_type = 1 means that there is no flux through the Northern wall. 
% BC_type = 2 means that there is no flux through the Southern wall.

% 1/8/94. Find bug in signs for original expressions. Hsurf is +ve INTO the sea. So
% the divergence of the lateral heat flux must equal - Hsurf. Change HsurfInteg to
% return the cube root of - the integral of Hsurf. Change this script in the following
% places : BCterm (for BC_type = 1), dHdy, and where Hsurf_a and Hsurf_b are set.

% 30/8/94. Looking at values of K. Find that the expression used in this script omitted
% the factor of gamma. Include this in the corrected script.

% Author : Tom, May '94.

clear
global BCterm y_zero

% Setup model parameters, and constants. (SI units throughout).

BC_type = 2 ;

gamma  = 0.3 ;
g      = 9.81 ;
c      = 4e3 ;
rhobar = 1024 ;
alpha  = 2e-4 ;
f      = 1e-4 ;
Nth    = 20 * f ;
H0     = 200  ;
seconds_in_a_year = 60*60*24*365 ;
seconds_in_a_day  = 60*60*24 ;

% MUST set Y_lo equal to zero, I think, or else the integrals won't work properly.
% Minimum value for No_pts is 20.
Y_lo   = 0 ;   
Y_hi   = 200.e3 ;
No_pts =  200 ;
y_zero = Y_lo ;

% Setup model variables. Require input of Hsurf(y), the surface heat flux in Wm^-2.
% This is provided by the Hsurf.m function. Must call Hsurf.m now, before the integration
% below, to setup global variables properly.

y_grid = [Y_lo : (Y_hi - Y_lo)/(No_pts-1) : Y_hi]  ;
const  = (f/Nth^2) * ((alpha*g)/(gamma*c*rhobar))^(1/3) ;
Hsurf1 = Hsurf(y_grid) ;
if (BC_type == 1)
   fprintf(1,'Boundary condition is no heat flux through Northern wall. \n\n') ;
   BCterm = - quad8('Hsurf',Y_lo,Y_hi) ;
   if (Hsurf(y_grid(No_pts)) ~= 0)
      fprintf(1,'WARNING : The surface heat flux is non-zero at the Northern wall. \n\n') ;
   end
elseif (BC_type == 2)
   fprintf(1,'Boundary condition is no heat flux through Southern wall. \n\n') ;
   BCterm = 0 ;
   if (Hsurf(y_grid(1)) ~= 0)
      fprintf(1,'WARNING : The surface heat flux is non-zero at the Southern wall. \n\n') ;
   end
else
   fprintf(1,'ERROR. Bad choice of boundary condition indicator; [%d].\n\n',BC_type) ;
   stop
end  

% For each value of y must calculate the Hsurf, integrated twice in the manner above.
% HsurfInteg calculates the cube root of the integral of Hsurf.

temp = zeros(1,No_pts) ;

for i = 1:No_pts
   temp(i)  = quad8('HsurfInteg',Y_lo,y_grid(i))  ;
end

H    = sqrt( 2 * const .* temp + H0^2) ;

% Now that H has been found as a funtion of y carry out various diagnostics to verify
% that the solution is correct, and to estimate the accuracy. The diagnostics are;
% i) Check that q = 0.
% ii) Check that heat is conserved by the solution by finding the error in the flux budget.
% iii) Find values of the mixed layer rho and Nmix, the diffusion coeff., K, the equivalent 
% subduction rate, S, the lateral flux of heat.
% iv) Find the value of the zonal velocity difference over the mixed layer.
% Many of these parameters require dHdy, which is calculated directly by integrating Hsurf
% once then dividing by H, rather than taking the finite differenced gradient of H. This
% should be much more accurate.

% Get dHdy. See hand written sheets for formula used.
dHdy  = zeros(1,No_pts) ;
for i = 1:No_pts
   dHdy(i) = const * ( -(quad8('Hsurf',Y_lo,y_grid(i)) -  BCterm)) ^ (1/3) ;
   dHdy(i) = dHdy(i) / H(i) ;
end

% Get rho. 
rho    = rhobar + (rhobar * Nth^2 / g).*H ;

% Get Nmix.
Nmix   = (Nth^2/f) .* dHdy ;

% Get K.
K      = gamma * (Nth^4/f^3) .*H.^2 .* dHdy.^2 ;

% Get S. Definition here is not identical to that used in John's paper because there z +ve
% means a different direction to H +ve in my model. Hence - sign inserted. See Abstract of
% paper.
S      = - alpha*g.*Hsurf1./(c*rhobar*Nth^2.*H) ;
S      = S*seconds_in_a_year ;

% Get Hlat.
Hlat   = ( (dHdy.^3) .* (H.^2) ) ./ const^3 ;

% Get error. Flux INTO domain through y=0 + Flux INTO sea - Flux OUT OF domain through y=Y_hi.
if (BC_type == 1)
   error  = H(1)*Hlat(1) + BCterm - H(No_pts)*Hlat(No_pts) 
else
   error =  H(1)*Hlat(1) - quad8('Hsurf',Y_lo,Y_hi) - H(No_pts)*Hlat(No_pts) 
end

% Get qmix.
qmix   =  f.*Nmix.^2./g - (Nth.^4).*(dHdy.^2)./(f*g) ;
max_q  =  max(qmix) 

% Calculate maximum velocity in mixed layer (LONM at surface). 
% Remove the 1st. or last entry depending whether or not there is no flux through the
% northen or southern wall (because velocity is zero there).
if (BC_type == 1)
   u0_mat = (Nth^2/f) .* H(1:No_pts-1).*dHdy(1:No_pts-1) ; 
   H_t    = H(1:No_pts-1) ;
   dHdy_t = dHdy(1:No_pts-1) ;
else
   u0_mat = (Nth^2/f) .* H(2:No_pts).*dHdy(2:No_pts) ; 
   H_t    = H(2:No_pts) ;
   dHdy_t = dHdy(2:No_pts) ;
end

% Calculate temperature field that eddy feels; vertical & horizontal.
% First get eddy width (from stone71.m script).

delta_mat = f.*H_t./u0_mat ;
Ri = 1 ;

growth_fastest = zeros(1,No_pts-1) ;
length_fastest = zeros(1,No_pts-1) ;

for i = 1:No_pts-1
  delta     = delta_mat(i) ;
  u0        = u0_mat(i) ;
  k_fastest = roots( [5.*delta.*delta./63 0 6.*(1 + Ri)./15 0 -1] ) ;
  k_fastest = max(real(k_fastest))  ;
  sigma_max = (k_fastest - (2.*k_fastest.^3/15).*                     ...
         (1 + Ri + 5.*delta.*delta.*k_fastest.*k_fastest./42) ) .* 1./(2*sqrt(3))  ;
  growth_fastest(i) = 1/(f*sigma_max) ;
  length_fastest(i) = 2*pi*u0/(f*k_fastest) ;
end

% Now get temperature contrast across the eddies and angle of isotherms in mixed 
% layer.
horiz_cont = (Nth^2/(alpha*g)).*dHdy_t.*length_fastest  ;
vert_cont  = (Nth^4/(alpha*g*f^2)).*(dHdy_t.^2).*H_t ;
angle      = (180/pi) .* atan(f^2./(Nth^2.*dHdy_t)) ;

% Work out the maximum slope of the mixed layer base, and the maximum allowed by the
% dynamics, based on the deformation radius scaling. See MIT lab book.
max_ml_slope = max(dHdy) ;
max_allowable_ml_slope = f/Nth ;
if (max_allowable_ml_slope < max_ml_slope)
   fprintf(1,'\n\n   WARNING !  Model mixed layer slopes too steeply ! \n\n');
end

% Work out magnitude of ignored term in the derivation for this model. It should be
% less than 1.
extra_term = max(dHdy.*Nmix./f) ;
if (extra_term > 0.5)
   fprintf(1, ...
   '\n\n   WARNING !  Inaccuracy greater than 50% in regions of steep mixed layers. \n\n') ;
end


% Print out ranges of variables ;
fprintf('[Maximum, Minimum] of \n\n') ;
fprintf('Eddy e-folding time (days)       [ %6.3f, %6.3f]  \n',  ...
max(growth_fastest)/seconds_in_a_day,min(growth_fastest)/seconds_in_a_day) ;
fprintf('Eddy length scale (km)           [ %6.3f, %6.3f]  \n', ...
 max(length_fastest)/1000,min(length_fastest)/1000) ;
fprintf('Vertical temp. contrast (K)      [ %6.3f, %6.3f]  \n',  ...
max(vert_cont),min(vert_cont)) ;
fprintf('Horizontal temp. contrast (K)    [ %6.3f, %6.3f]  \n',  ...
 max(horiz_cont),min(horiz_cont)) ;
fprintf('Zonal velocity contrast (m/s)    [ %6.3f, %6.3f]  \n',  ...
 max(u0_mat),min(u0_mat)) ;
fprintf('Isotherm angle to horizontal (o) [ %6.3f, %6.3f]  \n',  ...
 max(angle),min(angle)) ;
fprintf('\nMax mixed layer slope / max allowed mixed layer slope (o) [ %6.3f, %6.3f]  \n', ...
 (180/pi)*atan(max_ml_slope),(180/pi)*atan(max_allowable_ml_slope)) ;
fprintf('Maximum magnitude of ignored term (should be << 1)        [ %6.3f]  \n',  ...
 extra_term) ;

figure(1) ;
subplot(3,2,1) ;
plot(y_grid./1000,Hsurf1) ;
xlabel('Distance (km)') ;
ylabel('Hnet INTO sea (Wm^-2)') ;

subplot(3,2,2) ;
axis([Y_lo Y_hi -max(H) 0.0]) ;
plot(y_grid./1000,-H) ;
xlabel('Distance (km)') ;
ylabel('Mixed layer depth (m)') ;

subplot(3,2,3) ;
plot(y_grid./1000,rho-1000) ;
xlabel('Distance (km)') ;
ylabel('Mixed layer sigma (kgm^-3)') ;

subplot(3,2,4) ;
axis([ Y_lo Y_hi -max(K) 0.0]) ;
plot(y_grid./1000,K) ;
xlabel('Distance (km)') ;
ylabel('Diffusion coefficient (m^2s^-1)') ;

subplot(3,2,5) ;
axis([ Y_lo Y_hi  0.0 max(Nmix/f)]) ;
plot(y_grid./1000,Nmix./f) ;
xlabel('Distance (km)') ;
ylabel('Mixed layer N/f ()') ;

subplot(3,2,6) ;
plot(y_grid./1000,S) ;
xlabel('Distance (km)') ;
ylabel('Subduction rate (myr^-1)') ;

%subplot(4,2,7) ;
%plot(y_grid./1000,Hlat) ;
%xlabel('Distance (km)') ;
%ylabel('Lateral Heat Flux (Wm^-2)') ;

text1 = ['Thermocline N/f = ',num2str(Nth/f)] ;
gtext(text1)

% Print out diagnostics figure if user wants.
getkey = input('Hit <Y> to see diagnostics figure.   ','s') ;
if (getkey == 'Y' | getkey == 'y')
  temp = y_grid(1:length(y_grid)-1) ;
  figure(2)
  subplot(3,2,1) ;
  axis([ Y_lo Y_hi  0.0 max(growth_fastest./seconds_in_a_day)]) ;
  plot(temp./1000,growth_fastest./seconds_in_a_day) ;
  xlabel('Distance (km)') ;
  ylabel('Eddy e-folding time (days)') ;
  
  subplot(3,2,2) ;
  axis([ Y_lo Y_hi  0.0 max(length_fastest./1000)]) ;
  plot(temp./1000,length_fastest./1000) ;
  xlabel('Distance (km)') ;
  ylabel('Eddy length scale (km)') ;
  
  subplot(3,2,3) ;
  axis([ Y_lo Y_hi  0.0 max(vert_cont)]) ;
  plot(temp./1000,vert_cont) ;
  xlabel('Distance (km)') ;
  ylabel('Vertical temp. contrast (K)') ;
  
  subplot(3,2,4) ;
  axis([ Y_lo Y_hi  0.0 max(horiz_cont)]) ;
  plot(temp./1000,horiz_cont) ;
  xlabel('Distance (km)') ;
  ylabel('Horizontal temp. contrast (K)') ;

  subplot(3,2,5) ;
  axis([ Y_lo Y_hi  0.0 max(u0_mat)]) ;
  plot(temp./1000,u0_mat) ;
  xlabel('Distance (km)') ;
  ylabel('Veolcity contrast (m/s)') ;
  
  subplot(3,2,6) ;
  axis([ Y_lo Y_hi  0.0 max(angle)]) ;
  plot(temp./1000,angle) ;
  xlabel('Distance (km)') ;
  ylabel('Isotherm angle (deg)') ;
end


% Dump mixed layer characteristics to disk for use as diagnostics in CME.m
% Let the zonal extent be 20 deg. Note some of the parameters for this comparison (No_pts,
% y-spacing, no_x_pts etc.) are hard-coded in each script.
getkey = input('Hit <Y> to dump mixed layer chars to disk.  ','s') ;
if (getkey == 'Y' | getkey == 'y')

handle = fopen('Hnet3.output','w') ;
for i = 1:No_pts
    for j = 1:20
        fprintf(handle,'%g %g %12.6f %12.6f \n',j,i,H(i),rho(i)) ;
    end
end

fclose(handle) ;
end

% 27/7/94. Add code to test 3-D version of baroclinic eddy code (e.g.CME.m). Want to confirm
% Use form for flux from MIT lab book, which depends on rho and H (no assumption of smooth
% connection with thermocline below. This has 
% flux = ((gamma * c * g^2 ) / (alpha * f^3 * rhobar^2) ) .* drhody.^3 .* H.^2
% Compare this with Hlat, calculated above.
% Note that the fluxes calculated here are in W/m^2, and are integrated over the mixed layer
% depth, H, before the divergence is taken to find the effective vertical eddy flux.
% Then form vertical fluxes by multiplying by the mixed layer depth, and differentiating.
% In the plot of CME.m vertical flux use the limits from the Hnet3.m expression, to avoid
% high noise at edges due to 2nd derivative.

getkey =    ...
input('Hit <Y> to run comparison between different forms of flux expression.   ','s') ;
if (getkey == 'Y' | getkey == 'y')

   drhody = gradient(rho,y_grid) ;
   Hlat2 = ((gamma * c * g^2 ) / (alpha * f^3 * rhobar^2) ) .* drhody.^3 .* H.^2 ;

   Hsurf_a = -gradient((Hlat.*H),y_grid) ;
   Hsurf_b = -gradient((Hlat2.*H),y_grid) ;

   figure(3)
   subplot(2,2,1) ;
   axis([ Y_lo Y_hi  0.0 max(Hlat)]) ;
   plot(y_grid./1000,Hlat) ;
   xlabel('Distance (km)') ;
   ylabel('Flux from Hnet3 expression (W/m^-2)') ;

   subplot(2,2,2) ;
   axis([ Y_lo Y_hi  0.0 max(Hlat2)]) ;
   plot(y_grid./1000,Hlat2) ;
   xlabel('Distance (km)') ;
   ylabel('Flux from CME.m expression (W/m^-2)') ;

   subplot(2,2,3) ;
   axis([ Y_lo Y_hi  min(Hsurf_a) max(Hsurf_a)]) ;
   plot(y_grid./1000,Hsurf_a) ;
   xlabel('Distance (km)') ;
   ylabel('Vertical flux from Hnet3 expression (W/m^-2)') ;

   subplot(2,2,4) ;
   plot(y_grid./1000,Hsurf_b) ;
   axis([ Y_lo/1000 Y_hi/1000  min(Hsurf_a) max(Hsurf_a)]) ;
   xlabel('Distance (km)') ;
   ylabel('Vertical flux from CME.m expression (W/m^-2)') ;

end  

