% Script to calculate various parameters of interest in the mixed layer instability
% problem. 

% Author Tom, Sept '94. SI units throughout.

clear
close
close
close
close

% Range of independent variables.
H1      = 200  ;
H2      = 400  ;
H3      = 600  ;
Ri_lo   = 0.2 ;
Ri_hi   = 100 ;
drho_dy_lo = 1e-6 ;
drho_dy_hi = 1e-5 ;
No_pts  = 200 ;

Ri      = exp([log(Ri_lo):(log(Ri_hi)-log(Ri_lo))/No_pts:log(Ri_hi)]) ;
drho_dy = exp([log(drho_dy_lo):(log(drho_dy_hi)-log(drho_dy_lo))/No_pts:log(drho_dy_hi)]) ;
[Ri, drho_dy] = meshgrid(Ri,drho_dy) ;

% Setup script variables.
True  = 1;
False = 0 ;
secs_in_a_day = 3600*24 ;
PVU           = 1e-9 ;
Noverf_levels = [1,2,4,8,16,32,64,128,256] ;
PV_levels     = [0.001,0.01,0.05,0.1,0.2,0.4,0.8,1.6,3.2,6.4] ;
PV_levels     = [-PV_levels,0,PV_levels] ;
lambda_levels = [0.5,1,2,4,8,16,32,64,128,256] ;
timesc_levels = [0.25,0.5,0.75,1,1.5,2,3,5,10,20,50,100] ;
Eady_lambda_levels = [1,16,128] ;
Eady_timesc_levels = [0.5,1,2,5] ;
symm_timesc_levels = [0.1,0.25,0.5,1,2,5] ;
Noverf_title  = ['Mixed layer N/f'] ;
PV_title      = ['Mixed layer Potential Vorticity (PVU)'] ;
lambda1_title  = ['Instability wavelength (km), mixed layer depth = ',num2str(H1),'m.'] ;
lambda2_title  = ['Mixed layer depth = ',num2str(H2),' m.'] ;
lambda3_title  = ['Mixed layer depth = ',num2str(H3),' m.'] ;
timesc_title   = ['Instability timescale (days)'] ;
delta_ticks_t  = ['0.05';' 0.1';' 0.5';' 1  ';' 2  '] ;
delta_ticks    = str2num(delta_ticks_t) ;
delta_ticks_x  = ones(length(delta_ticks),1) .* Ri_hi .* 1.1 ;

% Setup physical model parameters
Radian     = 57.29578 ;
g      = 9.81 ;
Omega  = 7.292e-5 ;
rhobar = 1026 ;
alpha  = 2e-4 ;
latitude = 45/Radian ;

% Get Coriolis parameters.
f      = 2*Omega*sin(latitude) ;
f1     = 2*Omega*cos(latitude) ;
f1     = 0 ;


% Calculate everything.
N       = sqrt((g^2 * drho_dy.^2 .* Ri) ./ (f^2 * rhobar^2)) ;
drho_dz =  - rhobar .* N.^2 ./ g ;
PV      = - ((f1 + (g.*drho_dy)./(rhobar*f)).*drho_dy + f.*drho_dz) ./ rhobar ;
delta   = (f^2*rhobar)./(drho_dy.*g) ;
kmax    = sqrt( (6.3./delta.^2) .* ( -0.4 .* (1+Ri) + ...
                sqrt(0.16.*(1+Ri).^2 + 0.3175.*delta.^2) ) ) ;
sigmax  = (1/(2*sqrt(3))) * (kmax - (2.*kmax.^3./15).*(1+Ri+(0.119.*delta.^2.*kmax.^2))) ;
lambda1 = (2*pi*g*H1.*drho_dy)./(rhobar*f^2.*kmax) ;
lambda2 = (2*pi*g*H2.*drho_dy)./(rhobar*f^2.*kmax) ;
lambda3 = (2*pi*g*H3.*drho_dy)./(rhobar*f^2.*kmax) ;
timesc  = 1./(f.*sigmax) ;
delta_to_drho_dy = ((f^2*rhobar)./(delta_ticks.*g))' ;

% Calculate Eady timescale and wavelengths.
sigma_Eady = 0.3098*g.*drho_dy./(N .* rhobar) ;
k_Eady1     = 2*0.8031*f./(N.*H1) ;
k_Eady2     = 2*0.8031*f./(N.*H2) ;
k_Eady3     = 2*0.8031*f./(N.*H3) ;
lambda_Eady1= 2*pi./k_Eady1 ;
lambda_Eady2= 2*pi./k_Eady2 ;
lambda_Eady3= 2*pi./k_Eady3 ;
timesc_Eady =  1./sigma_Eady ;

% Calculate maximum symmetric instability growth rate.
sigma_symm = sqrt( ...
    ( Ri + delta.^2 - sqrt((Ri+delta.^2).^2 + 4*delta.^2.*(1-Ri))) ./ (2.*delta.^2)   ) ;
sigma_symm = imag(sigma_symm) ;
timesc_symm = 1./(f.*sigma_symm) ;

% Plot everything.
figure

% N/f plot.
subplot(3,1,1) ;
pic = contour(Ri,drho_dy,N./f,Noverf_levels) ;
set(gca,'YScale','log') ;
set(gca,'XScale','log') ;
title(Noverf_title) ;
xlabel('Richardson Number') ;
ylabel('Lateral density gradient (kg/m^4)') ; 
text(delta_ticks_x,delta_to_drho_dy,delta_ticks_t) ;
text(delta_ticks_x(1),drho_dy_hi*1.2,'delta') ;
clabel(pic,'manual') ;

% PV plot.
subplot(3,1,2) ;
pic = contour(Ri,drho_dy,PV./PVU,PV_levels) ;
set(gca,'YScale','log') ;
set(gca,'XScale','log') ;
title(PV_title) ;
xlabel('Richardson Number') ;
ylabel('Lateral density gradient (kg/m^4)') ; 
text(delta_ticks_x,delta_to_drho_dy,delta_ticks_t) ;
clabel(pic,'manual') ;

% Timescale plot.
subplot(3,1,3) ;
pic = contour(Ri,drho_dy,timesc./secs_in_a_day,timesc_levels) ;
set(gca,'YScale','log') ;
set(gca,'XScale','log') ;
title(timesc_title) ;
xlabel('Richardson Number') ;
ylabel('Lateral density gradient (kg/m^4)') ;
text(delta_ticks_x,delta_to_drho_dy,delta_ticks_t) ;
clabel(pic,'manual') ;
hold on
pic = contour(Ri,drho_dy,timesc_Eady./secs_in_a_day,Eady_timesc_levels,'--') ;
clabel(pic,'manual') ;
pic = contour(Ri,drho_dy,timesc_symm./secs_in_a_day,symm_timesc_levels,':') ;
clabel(pic,'manual') ;
hold off

% Adjust the whole figure.
set(gcf,'PaperPosition',[0.5 0.5 7.5 10]) ;


% Wavelength plot. Do 3 plots for each H chosen (H1, H2, H3).
figure
subplot(3,1,1) ;
pic = contour(Ri,drho_dy,lambda1./1000,lambda_levels) ;
set(gca,'YScale','log') ;
set(gca,'XScale','log') ;
title(lambda1_title) ;
xlabel('Richardson Number') ;
ylabel('Lateral density gradient (kg/m^4)') ; 
text(delta_ticks_x(1),drho_dy_hi*1.2,'delta') ;
text(delta_ticks_x,delta_to_drho_dy,delta_ticks_t) ;
clabel(pic,'manual') ;
hold on
pic = contour(Ri,drho_dy,lambda_Eady1./1000,Eady_lambda_levels,'--') ; 
clabel(pic,'manual') ;
hold off

subplot(3,1,2) ;
pic = contour(Ri,drho_dy,lambda2./1000,lambda_levels) ;
set(gca,'YScale','log') ;
set(gca,'XScale','log') ;
title(lambda2_title) ;
xlabel('Richardson Number') ;
ylabel('Lateral density gradient (kg/m^4)') ;
text(delta_ticks_x,delta_to_drho_dy,delta_ticks_t) ;
clabel(pic,'manual') ;
hold on
pic = contour(Ri,drho_dy,lambda_Eady2./1000,Eady_lambda_levels,'--') ;
clabel(pic,'manual') ;
hold off

subplot(3,1,3) ;
pic = contour(Ri,drho_dy,lambda3./1000,lambda_levels) ;
set(gca,'YScale','log') ;
set(gca,'XScale','log') ;
title(lambda3_title) ;
xlabel('Richardson Number') ;
ylabel('Lateral density gradient (kg/m^4)') ;
text(delta_ticks_x,delta_to_drho_dy,delta_ticks_t) ;
clabel(pic,'manual') ;
hold on
pic = contour(Ri,drho_dy,lambda_Eady3./1000,Eady_lambda_levels,'--') ;
clabel(pic,'manual') ;
hold off

% Adjust the whole figure.
set(gcf,'PaperPosition',[0.5 0.5 7.5 10]) ;
