% Script to make calculations based on the Stone '71 paper on Baroclinic stability under
% non-hydrostatic conditions.
% This code simply calculates various parameters based on the formulae in the paper. It also
% can display the fields of temperature, velocity and PV  of the calculation. User simply sets
% the important quantities of velocity scale, layer depth, PV and rotation rate. The ambient
% density, coefficient of expansion and gravity are set within the script itself.

% Author Tom, May 94. SI units throughout.

% Setup physical model parameters

g      = 10 ;
f      = 1e-4 ;
rhobar = 1026 ;
alpha  = 1e-4 ;

salinity = 35 ;

% Get values of model variables from user.

u0 = 0.50 ;
H  = 200 ;
q  = 0E-10 ;

% Setup internal model parameters which affect the manner of solution calculation  only.
% lo_wavel is the shortest wave instability which is marginally unstable. 

grid_pts = 50 ;
day_in_secs = 60*60*24 ;

polym = [ 5*delta*delta/42 0 Ri+1 0 -15/2 ] ;
lo_wavel = 0.9*max(real(roots( polym ) )) 
lo_wavel = 2*pi*u0/(f*lo_wavel) 
hi_wavel =100000 ;


% Calculate key model parameters from input data. This section also
% calculates the non-dimensional wavenumber and growth rate of the most unstable wave 
% (k_fastest & sigma_max). It then converts these into dimensional growth rates in days 
% (growth_fastest) and a length scale in m (length_fastest).

dTdz = (q + f*u0*u0/(g*H*H)) / (f*alpha) 
delta = f*H/u0 
Ri = alpha*g*H*H*dTdz/(u0*u0) 
Noverf = u0*sqrt(Ri)/(H*f)

k_fastest = roots( [5*delta*delta/63 0 6*(1 + Ri)/15 0 -1] ) ;
k_fastest = max(real(k_fastest)) 
sigma_max = (k_fastest - (2*k_fastest^3/15)*(1 + Ri + 5*delta*delta*k_fastest*k_fastest./42) ) ...
             * 1/(2*sqrt(3)) 
growth_fastest = 1/(day_in_secs*f*sigma_max)
length_fastest = 2*pi*u0/(f*k_fastest)


% Calculate & display fields of temperature, velocity and PV.






% Calculate the dispersion curve for the instability & display.
% k and sigma are the non-dimensionalised wavenumber and growth rate of instabilities. The
% plot converts them back into a dimensional plot of e-folding scale (days) and wavelength (m).
% The range of wavelengths considered corresponds to lo_wavel and hi_wavel (dimensional).

k = 2*pi*[ 1/hi_wavel : (1/lo_wavel - 1/hi_wavel)/grid_pts : 1/lo_wavel ] ;
k = u0.*k./f   ;
sigma = (k - (2.*k.^3./15).*(1 + Ri + 5*delta*delta.*k.*k./42) ) .* 1./(2*sqrt(3))  ;

close
subplot(2,1,1) ;
plot(k,sigma) ;
xlabel('Wavenumber ()')
ylabel('Growth Rate ()')
subplot(2,1,2) ;
axis([lo_wavel hi_wavel 0 max(1./(day_in_secs*f.*sigma)) ]) ;
plot(2*pi*u0./(f.*k),1./(day_in_secs*f.*sigma)) ;
xlabel('Wavelength (m)')
ylabel('Timescale (days)')

