% Script to read, filter and display tsg data; specifically the gradients in tsg density.
% tsg data has the following properties;
% i) some bad data points (-999.0),
% ii) periods when the ship is moving very slowly,
% iii) interruptions in the data stream (large jumps in distance run variable).

% Data source is from pstar tsg files. Run peos83 to get density (sigt83), then pavrge
% to average the data into 250m bins (can't find another easy way to grid the data),
% Finally, run mlist the distrun and sigma variables to ascii file. Remove header with
% vi, then the file is ready to load directly into matlab.

% Add code to plot a 2nd figure with the equivalent diffusion coefficients histograms
% graphed, as a function of different mixed layer depths.

% Author : twnh, aug '94.

tic
close all

bad_data   = -999. ;
spacing    = 3 ;
wave_cut   =  600 ;
cut_off    = 20 ;

depths = [50,100,200,400,800] ;

gamma   = 0.10 ;
g       = 9.81 ;
rho_bar = 1026 ;
f       = 1e-4 ;

if(exist('raw_data') == 0)
  file = input(' Please enter the ascii tsg file you wish to process  :  ','s') ;
  raw_data = read(file,'%g %g %g',1,-1,1) ;
end

name = input(' Please enter the name of the data source (appears on plots) : ','s') ;

% Extract raw_x and raw_rho from the data. Convert x to SI from km
raw_x   = raw_data(:,2)*1000 ; ;
raw_rho = raw_data(:,3) ;
raw_no  = size(raw_data,2) ;

% Get mean record separation.
mean_sep = mean(diff(raw_x)) ;
lo_spacing = mean_sep / spacing ;
%hi_spacing = mean_sep * spacing ;
Nyquist = 1/(2*mean_sep) ;

% Check to see if filter will work OK.
if(wave_cut < 1/Nyquist) 
  fprintf(1,' ERROR !  wave_cut [%g] is too low. It should be >= [%g].',wave_cut, 1/Nyquist) ;
  stop ;
end

% Process data to remove bad data, and those data with odd spacing.
 
fprintf(1,' Removing bad_data ... ') ;
indices = (raw_rho == bad_data) ;
new_rho = raw_rho(~indices) ;
new_x   = raw_x(~indices) ;
fprintf(1,' removed %g data. \n',sum(indices)) ;

fprintf(1,' Removing outliers ... ') ;
dist    = [0;diff(new_x)] ;
%indices = (dist > hi_spacing | dist < lo_spacing) ;
indices = (dist < lo_spacing) ;
new_rho = new_rho(~indices) ;
new_x   = new_x(~indices) ;
fprintf(1,' removed %g data.\n',sum(indices)) ;

% Generate Butterworth filter based on wave_cut, the cutoff wavelength.
fprintf(1,' Filtering data with low-pass filter of wavelength %g m ... ',wave_cut) ;
mean_sep = mean(diff(new_x)) ;
[b,a] = butter(7,1/(Nyquist*wave_cut)) ;
filtered_rho = filtfilt(b,a,new_rho) ;
fprintf(1,' done.\n') ;

% Get normalised auto correlation sequence.
fprintf(1,' Calculating auto covariance ... ') ;
auto_cov = xcov(filtered_rho,'coeff') ;
auto_cov = auto_cov([size(auto_cov)/2:size(auto_cov)]) ;
auto_cov_x = mean_sep .* [1:size(auto_cov)] ;
% Get zero-crossing.
for i = 1:size(auto_cov)
   if(auto_cov(i)<0)
      break ;
   end
end
zero_cross = auto_cov_x(i) 
fprintf(1,' done.\n') ;

% Do histogram of density gradients. 
fprintf(1,' Calculating normalised histogram of density gradients ... ') ;
grad_rho = abs(gradient(filtered_rho',mean_sep)) ;
[n,x]    = hist(grad_rho,50) ;
% Remove bins with less than cut_off hits in them.
  [toti,totj] = find(n < cut_off) ;
  n(totj) = [] ;
  x(totj) = [] ;

n = n./size(filtered_rho,1) ;
fprintf(1,' done.\n') ;
subplot(2,1,1) ;
%plot(x,n,'+') ;
%subplot(2,1,2) ;
semilogy(x,n,'+') ;
hold off
xlabel('Lateral density gradient, kg/m^4') ;
ylabel('Normalised frequency') ;
text = sprintf('Histogram of density gradients from %s, with low pass filter of cutoff %d m.',name,wave_cut) ;
title(text) ;

% Do fft and graph. Pad the real data with zeros to the next highest power of 2.
% Use signal processing toolbox to find power spectrum with spectrum fn. and specplot.
wavelengths = ['400';'200';'100';' 40';' 20';' 10';' 4 ';' 2 ';' 1 '] ;
fprintf(1,' Calculating PSD of data ... ') ;
p = spectrum(filtered_rho,256) ;
fprintf(1,' done.\n') ;
%figure
subplot(2,1,2)
specplot(p,Nyquist*2) ;
set(gca,'XScale','log') ;
set(gca,'XTickLabelMode','manual') ;
set(gca,'XTickLabels',wavelengths) ;
set(gca,'XTickMode','manual') ;
set(gca,'XTick',1./(1000*str2num(wavelengths))) ;
xlabel('Wavelength (km)') ;

set(gcf,'PaperPosition',[0.5 0.5 7.5 10]) ;


%n = log(size(filtered_rho,1))/log(2) ;
%n = 2^ceil(n) ;
%n = 1024 ;
%transform = fft(filtered_rho,n) ;
%transform = transform .* conj(transform) / n ;
%freq = [0:2*Nyquist/(n-1):Nyquist]' ;
%transform((n/2) + 1 : n) = [] ;
%transform(2:n/2) = 2.*transform(2:n/2) ;
%figure
%semilogy(freq,transform) ;
%figure
%semilogy(1./freq,transform) ;

% Do series of plots showing the effective diffusion coefficient for a few (constant)
% mixed layer depths.

figure
for i = 1:length(depths)
   %subplot(length(depths),1,i)
   K = ((gamma * g^2) / (rho_bar^2 * f^3) ) .* depths(i)^2 .* x.^2 ;
   %semilogy(K,n) ;
   loglog(K,n) ;
   hold on
end
loglog([250,250],[1e-4,1],'r') ;
axis([1 1e5 1e-4 1]) ;
ylabel('Normalised frequency') ;
xlabel('Diffusion coefficient (m^2/s)') ;
title('Histogram of diffusion coefficients for different mixed layer depths') ;
set(get(gca,'Title'), 'FontSize',18 );
set(get(gca,'Xlabel'),'FontSize',14) ;
set(get(gca,'Ylabel'),'FontSize',14) ;
grid on

figure
semilogy(x,n) ;
hold off
xlabel('Lateral density gradient, kg/m^4') ;
ylabel('Normalised frequency') ;
text = sprintf('Histogram of density gradients from %s.',name) ;
title(text) ;
set(get(gca,'Title'), 'FontSize',18 );
%axis([1 1e5 1e-4 1]) ;
grid on
set(get(gca,'Xlabel'),'FontSize',14) ;
set(get(gca,'Ylabel'),'FontSize',14) ;

figure
%subplot(2,1,1)
plot(new_x,new_rho)
ylabel('Filtered density') ;
%subplot(2,1,2)
%plot(auto_cov_x, auto_cov) ;
%ylabel('Auto covariance') ;
toc
