% Script to run time series diagnostics on the CM5 channel model output. See diagnostic.m
% for diagnostics at a given time. 
% This function compares the model mixed layer evolution with the 1-D prediction, and
% also calculates spectra of the length scales in the model mixed layer. These can be used
% to diagnose the instability time scale.
%
% Author twnh Nov '94.

close all
clear 
clear global

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Get input parameters and setup variables.                                    %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

global x_spacing y_spacing z_spacing this_path NX NY NZ
global x_domain y_domain z_domain
global p_x_grid p_y_grid p_z_grid
global u v theta q E ang_mom trac z_vort secs_in_a_day
global f g c N2 alpha rho_bar flux time Half_flux Initial_temps cooling_width cooling_offset

% Constants. SI units unless explicitly stated otherwise. NB variables with dimensions of
% time in this script are generally in days. This applies to ; times and tinst.
Earth_rad  = 6370e3 ;
Radian     = 57.29578 ;
g          = 9.81 ;
c          = 4e3 ;
rho_bar    = 1000 ;
True       = 1 ;
False      = 0 ;
secs_in_a_day = 60*60*24 ;
Init_temp  = 12 ;
surface_layer = 19 ;

READ_DUMP = 0 ;

% Read in list of parameters for this data.
param_file_name =   ...
   input(' Please enter the name of the m-file with the parameters for this run : ','s') ;
time_series =       ...
   input(' Please enter the name of the data file for this time series analysis : ','s') ;
feval(param_file_name) ;

% Loop over entries in the time_series data files, calculating the various quantities
% required.
fid = fopen(time_series) ;
[dirs,cnt] = fscanf(fid,'%g')   ;
fclose(fid) ;
% times is in units of days.
%times   = [0;dirs]./(24*60)  ;
times   = dirs./(24*60)  ;

% Define grids.
setup_grids(x_spacing,y_spacing,z_spacing) ;

% Surface forcing. See notes in white lab book about definition of flux used (23/1/95);
% it has several peculiar and unexpected features.
flux     = Half_flux.*(tanh((2.*(p_y_grid +125 - cooling_offset))./cooling_width) + 1 ) ;
flux(1:9) = zeros(size(1:9)) ;
flux(NY)  = 0 ;
flux(1)   = 0 ;
flux     = [p_y_grid',flux'] ;

main_dir = this_path ;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Begin loop over model dumps.                                                 %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% Begin loop.
for i = 1:size(dirs,1)
   this_path = [main_dir,'/',int2str(dirs(i))] ;
   fprintf(1,' \nRunning on directory %s ... ',this_path) ;

% Try to read in  variables from external file; if no file exists read
% in the full model dumps in order.
   temp_file = [main_dir,'/summary/',int2str(dirs(i))] ;
   fid = fopen(temp_file,'r') ;
   if(fid ~= -1)
%  Read in external dump file directly.
      READ_DUMP = 1 ;
      fprintf(1,' Reading dump file from previous calculation. ') ;
      mean_theta         = fread(fid,[NZ,NY], 'float32') ;
      theta_zero_cross_t = fread(fid,1,       'float32') ;
      v_zero_cross_t     = fread(fid,1,       'float32') ;                    
      length_t           = fread(fid,1,       'float32') ;                   
      max_temp_t         = fread(fid,1,       'float32') ;                   
      min_temp_t         = fread(fid,1,       'float32') ;                   
      mean_temp_t        = fread(fid,1,       'float32') ;                   
      one_D_H_t          = fread(fid,1,       'float32') ;                   
      one_D_temp_t       = fread(fid,1,       'float32') ;                   
      dtheta_dy_t        = fread(fid,1,       'float32') ;                   
      Nhalf_t            = fread(fid,1,       'float32') ;                   
      Shalf_t            = fread(fid,1,       'float32') ;                   
      Nhalf_lost_t       = fread(fid,1,       'float32') ;                   
      Shalf_lost_t       = fread(fid,1,       'float32') ;                   
      fclose(fid) ;

% Find observed mixed layer stratification.
% Change format of arrays to conform with d_dn requirements.
      old_NX = NX ;
      NX = 1 ;
      temp_theta        = reshape(mean_theta',NY*NZ,1) ;
      N2obs = (g*alpha).*d_dn(temp_theta,'z',p_z_grid,z_domain) ;
      N2obs = reshape(N2obs,NY,NZ)' ;
      NX = old_NX ;
      temp = [] ;
      for j = NZ:-1:1
         if (p_z_grid(j) > -sqrt(2)*one_D_H_t) 
            temp = [temp;mean(N2obs(j,[floor(NY*1/3):floor(NY*2/3)+1]))] ;
         end
      end
      temp2 = find(isnan(temp)) ; 
      temp(temp2) = [] ;
      if (isempty(temp))
         mean_N2 = [mean_N2;N2] ;
      else
         mean_N2 = [mean_N2;mean(temp)] ;
      end
% Replace any -ve N2 with zero.
      %mean_N2 = (mean_N2 + abs(mean_N2))./2 ;

   else
%  External dump file not found; read in full data files.
      fprintf(1,' Reading original model fields. \n') ;

% Load theta, v.
      theta = readmodel('theta.sun.b',this_path,'float32') ;
      v     = readmodel('v.sun.b',this_path,'float32') ;

% Remove land.
      if (i == 1)
         mask1 = find(abs(theta) < 5) ;
         templ = NaN.*ones(size(mask1)) ;
      end

      theta(mask1) = templ ;
      v(mask1)     = templ ;

      if( i == 1 )
         mask = ~isnan(theta) ;
      end

% Calculate along channel average of theta field.
      mean_theta = meanyz(theta,x_spacing,mask) ;

% Get xy slice of temp and meridional flow.
      theta_slice = getxy(theta,surface_layer) ;
      v_slice     = getxy(v,surface_layer) ;

% Calculate the auto-correlations of theta and v along the channel centre near
% the surface (layer 'surface_layer').
      surf_cent_theta = mean([theta_slice(floor(NY/2),:);theta_slice(floor(NY/2)+1,:)]) ;
      surf_cent_v     = mean([v_slice(floor(NY/2),:);v_slice(floor(NY/2)+1,:)]) ;
      theta_prime     = [theta_prime;surf_cent_theta] ;   
      v_prime         = [v_prime;surf_cent_v] ;   
      theta_corr_t    = xcov(surf_cent_theta([1:floor(NX/2)]),'coeff') ;
      theta_corr_t    = flipud(theta_corr_t(1:floor(NX/2)))' ;
      v_corr_t        = xcov(surf_cent_v([1:floor(NX/2)]),'coeff') ;
      v_corr_t        = flipud(v_corr_t(1:floor(NX/2)))' ;

% Find the zero crossing point of the theta and v auto correlation series.
      for j = 1:100
         if(theta_corr_t(j) < 0)
            break ;
         end
      end
      theta_zero_cross_t = j ;
      for j = 1:100
         if(v_corr_t(j) < 0)
            break ;
         end
      end
      v_zero_cross_t = j ;

% Get the maximum, minimum and mean temperatures along the channel centre at the 
% surface_layer. Also find the 1D depth & temperature prediction.
      max_temp_t   = max(surf_cent_theta) ;
      min_temp_t   = min(surf_cent_theta) ;
      mean_temp_t  = mean(surf_cent_theta) ;
      one_D_H_t    = -mixed_layer(mean(flux([floor(NY/2),floor(NY/2)+1],2)),N2,times(i)*secs_in_a_day)  ;
      one_D_temp_t = Init_temp - (N2/(alpha*g)) * one_D_H_t ;

%  Get gradient  in temperature at the centre of the channel. (Answer in gridpoints).
%  dtheta_dy_t = mean(theta_slice(floor(NY/2)-1,:) - theta_slice(floor(NY/2)+1,:)) / ...
               (p_y_grid(floor(NY/2)-1) - p_y_grid(floor(NY/2)+1))   ;
% Alternative definition of the gradient based on difference across zone of cooling.
% See below (after loop) for another definition of dtheta_dy from the 1D law.
      dtheta_dy_t = mean(theta_slice(floor(NY/3)-1,:) - theta_slice(floor(NY*2/3)+1,:)) /  ...
                  (p_y_grid(floor(NY/3)-1) - p_y_grid(floor(NY*2/3)+1))   ;
      length_t = 2*std(theta_slice(floor(NY/2),:)) / dtheta_dy_t ;

% Integrate the temperature at time = 0, to budget heat later on. Do the job here
% rather than earlier, because we need to know the mask from a real theta field.
      if( i == 1)
         old_NX = NX ;
         NX = 1 ;

         mean_mask          = ~isnan(mean_theta)  ;
         mean_mask_2        = reshape(mean_mask',NY*NZ,1) ;

         [t,Init_theta]     = meshgrid([1:NY],fliplr(Initial_temps)) ;
         dummy              = zeros(NZ,floor(NY/2)) ;

        
         Shalf_theta        = [Init_theta(:,[1:floor(NY/2)]),dummy] ;
         Nhalf_theta        = [dummy,Init_theta(:,[floor(NY/2)+1:NY])] ;

%     Change format of arrays to conform with volume_integ requirements.
         Shalf_theta        = reshape(Shalf_theta',NY*NZ,1) ;
         Nhalf_theta        = reshape(Nhalf_theta',NY*NZ,1) ;
         Init_theta         = reshape(Init_theta',NY*NZ,1) ;

         Init_Shalf_heat    = volume_integ(Shalf_theta,x_domain,y_spacing,z_spacing, ...
                          mean_mask_2)*c*rho_bar ;
         Init_Nhalf_heat    = volume_integ(Nhalf_theta,x_domain,y_spacing,z_spacing, ...
                          mean_mask_2)*c*rho_bar  ;
         Init_channel_heat  = volume_integ(Init_theta,x_domain,y_spacing,z_spacing, ...
                          mean_mask_2)*c*rho_bar  ;

% Check calculation.
         if (                              ...
          abs((Init_Shalf_heat + Init_Nhalf_heat - Init_channel_heat) /    ...
              Init_channel_heat) > 1e4*eps)
            fprintf(1,                     ...
            ' WARNING : Initial North + South channel halves do not add up. \n\n') ;
         end

         clear Init_theta t
         NX = old_NX ;
       end

% Calculate heat in southern and northern halves of channel, and add interfacial
% heat loss. Use the mean theta section (averaged along channel).
      old_NX = NX ;
      NX = 1 ;

      Shalf_theta    = [mean_theta(:,[1:floor(NY/2)]),dummy] ;
      Nhalf_theta    = [dummy,mean_theta(:,[floor(NY/2)+1:NY])] ;
   
%  Change format of arrays to conform with volume_integ requirements.
      mean_theta_2   = reshape(mean_theta',NY*NZ,1) ;
      Shalf_theta    = reshape(Shalf_theta',NY*NZ,1) ;
      Nhalf_theta    = reshape(Nhalf_theta',NY*NZ,1) ;

      Shalf_heat     = volume_integ(Shalf_theta,x_domain,y_spacing,z_spacing, ...
                       mean_mask_2)*c*rho_bar  ;
      Nhalf_heat     = volume_integ(Nhalf_theta,x_domain,y_spacing,z_spacing, ...
                       mean_mask_2)*c*rho_bar  ;
      channel_heat   = volume_integ(mean_theta_2,x_domain,y_spacing,z_spacing, ...
                       mean_mask_2)*c*rho_bar  ;
      clear mean_theta_2 theta
      NX = old_NX ;

% Check calculation.
         if (                              ...
          abs((Shalf_heat + Nhalf_heat - channel_heat) /    ...
              channel_heat) > 1e4*eps)
            fprintf(1,                     ...
            ' WARNING : North + South channel halves do not add up. \n\n') ;
         end

      Shalf_loss  = mean(flux([2:floor(NY/2)  ],2)) * NX * (NY/2) * mean(x_spacing) *   ...
                  mean(y_spacing) * times(i)*secs_in_a_day  ;
      Nhalf_loss  = mean(flux([floor(NY/2)+1:NY],2)) * NX * (NY/2) * mean(x_spacing) * ...
                  mean(y_spacing) * times(i)*secs_in_a_day  ;
      channel_loss = Shalf_loss + Nhalf_loss ;
     

% Display the ratio of the heat budget error to floating point precision.
%      fprintf(1,' Precision of heat budget [%d] times floating point. \n',    ...
%              (Init_channel_heat - channel_loss - channel_heat) / (channel_heat*eps) ) ;
% Update variables.
      Nhalf_t = Nhalf_heat-Init_Nhalf_heat ;
      Shalf_t = Shalf_heat-Init_Shalf_heat ;
      Nhalf_lost_t = -Nhalf_loss ;
      Shalf_lost_t = -Shalf_loss ;


% Save a selection of variables to an external matlab file for later use.
      temp_file = [main_dir,'/summary/',int2str(dirs(i))] ;
      fid = fopen(temp_file,'w') ;
      if(fid == -1)
         fprintf(1,' ERROR : Opening file [%s]. \n\n',temp_file) ;
         return
      end
         %fprintf(1,' Saving data to disk. \n') ;
         fwrite(fid,mean_theta,                 'float32') ;
         fwrite(fid,theta_zero_cross_t,         'float32') ;
         fwrite(fid,v_zero_cross_t,             'float32') ;
         fwrite(fid,length_t,                   'float32') ;
         fwrite(fid,max_temp_t,                 'float32') ;
         fwrite(fid,min_temp_t,                 'float32') ;
         fwrite(fid,mean_temp_t,                'float32') ;
         fwrite(fid,one_D_H_t,                  'float32') ;
         fwrite(fid,one_D_temp_t,               'float32') ;
         fwrite(fid,dtheta_dy_t,                'float32') ;
         fwrite(fid,Nhalf_t,                    'float32') ;
         fwrite(fid,Shalf_t,                    'float32') ;
         fwrite(fid,Nhalf_lost_t,               'float32') ;
         fwrite(fid,Shalf_lost_t,               'float32') ;
         fclose(fid) ;

   end

% T_prime; the average difference in temperature across the front. This is not
% saved by the code above; so calculate it from the mean theta section.
   T_prime_t           = mean(mean_theta(surface_layer , 2:floor(NY/3)-1)) -  ...
                         mean(mean_theta(surface_layer , floor(NY*2/3)+1:NY-1)) ;

% Alternative definition of T_prime; based on the mean difference in temp across
% the front, averaged over the oneD layer depth.
   if(1)
       T_prime_t = [] ;
       temp = [] ;
       for j = NZ:-1:1
          if (p_z_grid(j) > -sqrt(2).*one_D_H_t)
             temp = [temp; ...
             mean(mean_theta(j,[2:floor(NY/3)-1])) - ...
             mean(mean_theta(j,[floor(NY*2/3)+1:NY-1]))] ;
          end
       end
       temp2 = find(isnan(temp)) ;
       temp(temp2) = [] ;
       if (isempty(temp))
          T_prime_t = [T_prime_t;0] ;
       else
          T_prime_t = [T_prime_t;mean(temp)] ;
       end
       T_prime_t = mean(T_prime_t) ;
   end
   
% Update timeseries matrices.
   theta_corr          = [theta_corr;theta_corr_t] ;
   v_corr              = [v_corr;v_corr_t] ;
   theta_zero_cross    = [theta_zero_cross;theta_zero_cross_t] ;
   v_zero_cross        = [v_zero_cross;v_zero_cross_t] ;
   max_temp            = [max_temp;max_temp_t] ;
   min_temp            = [min_temp;min_temp_t] ;
   mean_temp           = [mean_temp;mean_temp_t] ;
   one_D_H             = [one_D_H;one_D_H_t] ;
   one_D_temp          = [one_D_temp;one_D_temp_t] ;
   length              = [length; length_t] ;
   dtheta_dy           = [dtheta_dy;dtheta_dy_t] ;
   Nhalf               = [Nhalf;Nhalf_t]  ;
   Shalf               = [Shalf;Shalf_t] ;
   Nhalf_lost          = [Nhalf_lost;Nhalf_lost_t]  ;
   Shalf_lost          = [Shalf_lost;Shalf_lost_t] ;
   T_prime             = [T_prime; T_prime_t] ;

end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Time series analysis complete; calculate diagnostic parameters.              %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Calculate various parameters from timeseries.
% Heat budgets; based on summing the 2 channel halfs, and integrating the whole
% channel. These two should be identical and very small (roundoff error only if
% the model run was using the 'New flux' DiffuType). Expts 1-4 and the first few
% days of 5 use an older algorithm called 'CRAY model', which is not 100% 
% conservative. 
% N.B. +ve sum means the model is GAINing heat.

channel      = Nhalf+Shalf ;
channel_lost = Nhalf_lost + Shalf_lost ;
sum          = Shalf+Nhalf-Shalf_lost-Nhalf_lost   ;
E            = Shalf - Shalf_lost ;

%if( tinst_type == 1)
% Instability timescale; definition based on the time when the southern half of the
% channel loses as much heat sideways to the north as it does through the surface.
   %tinst        = interp1((Shalf-sum)./(Shalf_lost-sum),times,2,'spline') ;

%elseif (tinst_type == 2)
% Definition based on some fraction of the heat flux difference between the 
% channel halves.
   epsilon = 0.20 ;
   delta_E =  - log(cosh(3))*epsilon*Half_flux*x_domain*cooling_width ;

% Fit spline through E(t)/t.
   pp = csapi(E./(secs_in_a_day.*times),(secs_in_a_day.*times)) ;
   tinst = fnval(pp,delta_E)./secs_in_a_day ;
%end

if( tinst > times(i))
   fprintf(' WARNING !  There are not enough timesteps to reach tinst.\n\n') ;
end

% Code to only consider times up to tinst in the fit below.
cnt = 1 ;
for j = 1:size(dirs,1)
   if(times(j)<tinst)
      cnt = cnt + 1 ;
   end
end

% Truncate each vector to tinst.
times([cnt+1:j])      = [] ;
one_D_H([cnt+1:j])    = [] ;
E([cnt+1:j])          = [] ;
T_prime([cnt+1:j])    = [] ;
dtheta_dy([cnt+1:j])  = [] ;
mean_N2([cnt+1:j])    = [] ;
Shalf([cnt+1:j])      = [] ;
Shalf_lost([cnt+1:j]) = [] ;
Nhalf_lost([cnt+1:j]) = [] ;
Nhalf([cnt+1:j])      = [] ;
sum([cnt+1:j])        = [] ;
i = cnt ;


% Get Nmix; mixed layer stratification for zero PV.
Nmix = (g*alpha/(f*cooling_width))*T_prime/sqrt(N2) ;

% Print out the actual N2 observed from the model at tinst and the prediction
% from the simple scaling. Nml/Nth ~ Nth.f/Ty
pp = csapi(times, T_prime./cooling_width) ;
T_y = fnval(pp,tinst)  ;
pp = csapi(times, mean_N2) ;
Ntemp = sqrt(fnval(pp,tinst)/N2)  ;
pp = csapi(times,Nmix) ;
Nmixtemp = fnval(pp,tinst) ; 

% Values of N2 compared.
Ntemp/(sqrt(N2)*f/(T_y*alpha*g)) ;
Nmixtemp*sqrt(N2) ;
Ntemp*sqrt(N2) ;
sqrt(N2) ;

% Choose two estimates of dtheta_dy. First from the model fields themselves, I.e.
% from definition above. Second, from 1D law (see white lab book, 8/2/95).
% Try third definition of dtheta_dy = T_prime / (zone_width) ;
zone_width = cooling_width*2 ;
dtheta_dy1  = dtheta_dy ;
dtheta_dy2  = 2.*sqrt((2.*times.*secs_in_a_day.*Half_flux.*N2)./(g*alpha*rho_bar*c))./cooling_width ;
dtheta_dy3  = T_prime / (zone_width) ;
dtheta_dy   = dtheta_dy3  ;

wanna_plot = 0 ;

% Gamma; heat flux parameterisation. Choose from several types. 
[best_gamma1,flux1] = param_G(E, times, T_prime, one_D_H, tinst, zone_width, wanna_plot) ;
[best_gamma2,flux2] = param_S(E, times, T_prime, one_D_H, tinst, zone_width, wanna_plot) ;
[best_gamma6,flux6] = param_NI(E, times, T_prime, one_D_H, tinst, zone_width, wanna_plot) ;

fprintf(1,'\n\n Results of parameterisations : (best values for empirical constant) \n') ;
fprintf(1,' Green   :  [%f], \n Stone   :  [%f], \n New     :  [%f].\n\n',best_gamma1, best_gamma2, ...
        best_gamma6) ;

flux = flux6 ;
best_gamma = best_gamma6 ;

fit = Shalf_lost-flux ;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Display output.                                                              %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Plot of T, v, and the T autocorelation along the channel, at the channel centre and
% different times.
if( READ_DUMP ~= 1)
   auto_correl_fig(i,p_x_grid,theta_prime,one_D_temp,times,theta_corr, ...
                   param_file_name,v_prime,v_corr) ;
end

% Do a plot of zero crossing vs time, mean temp vs time, max temp vs time, std dev
% temp vs time.

%zero_cross_fig(times, theta_zero_cross, v_zero_cross, length, i, param_file_name, ...
               %max_temp, min_temp, mean_temp, one_D_temp) ;

% Draw a figure of heat budget.
heat_budget_fig(i, times, Nhalf, Shalf, sum, param_file_name, ...
                Shalf_lost, Nhalf_lost, tinst, best_gamma, fit) ;
best_gamma
tinst
