% Script to run time series diagnostics on the CM5 channel model output. 
% This version reads in dump files created by the stand-alone diagnostic2
% script to fit parameterisations to all the expts run.
%
% Author twnh Nov '95.

function answers = inter_diagnostic3(expt, expt_times, const_flag)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Get input parameters and setup variables.                                    %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

global x_spacing y_spacing z_spacing this_path NX NY NZ
global x_domain y_domain z_domain
global p_x_grid p_y_grid p_z_grid
global u v theta q E ang_mom trac z_vort secs_in_a_day
global f g c N2 alpha rho_bar flux time Half_flux Initial_temps cooling_width cooling_offset

% Constants. SI units unless explicitly stated otherwise. NB variables with dimensions of
% time in this script are generally in days. This applies to ; times and tinst.
Earth_rad  = 6370e3 ;
Radian     = 57.29578 ;
g          = 9.81 ;
c          = 4e3 ;
rho_bar    = 1000 ;
secs_in_a_day = 60*60*24 ;

% Data from input args to function.
param_file_name = expt ;
time_series = expt_times ;
feval(param_file_name) ;

% Loop over entries in the time_series data files, calculating the various quantities
% required.
fid = fopen(time_series) ;
[dirs,cnt] = fscanf(fid,'%g')   ;
fclose(fid) ;
% times is in units of days.
times   = dirs./(24*60)  ;

% Define grids.
setup_grids(x_spacing,y_spacing,z_spacing) ;

main_dir = this_path ;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Begin loop over model dumps.                                                 %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% Begin loop.
for i = 1:size(dirs,1)
   this_path = [main_dir,'/',int2str(dirs(i))] ;
   fprintf(1,' \nRunning on directory %s ... ',this_path) ;

   temp_file = [main_dir,'/summary/',int2str(dirs(i))] ;
   fid = fopen(temp_file,'r') ;

   if(fid ~= -1)
%  Read in external dump file directly.
      fprintf(1,' Reading dump file from previous calculation. ') ;
      mean_theta         = fread(fid,[NZ,NY], 'float32') ;
      theta_zero_cross_t = fread(fid,1,       'float32') ;
      v_zero_cross_t     = fread(fid,1,       'float32') ;                    
      length_t           = fread(fid,1,       'float32') ;                   
      max_temp_t         = fread(fid,1,       'float32') ;                   
      min_temp_t         = fread(fid,1,       'float32') ;                   
      mean_temp_t        = fread(fid,1,       'float32') ;                   
      one_D_H_t          = fread(fid,1,       'float32') ;                   
      one_D_temp_t       = fread(fid,1,       'float32') ;                   
      dtheta_dy_t        = fread(fid,1,       'float32') ;                   
      Nhalf_t            = fread(fid,1,       'float32') ;                   
      Shalf_t            = fread(fid,1,       'float32') ;                   
      Nhalf_lost_t       = fread(fid,1,       'float32') ;                   
      Shalf_lost_t       = fread(fid,1,       'float32') ;                   
   else
      fprintf(1,' ERROR opening file [%s].',temp_file) ;
      return
   end
      fclose(fid) ;

% Definition of T_prime; based on the mean difference in temp across
% the front, averaged over the oneD layer depth.
   if(1)
       T_prime_t = [] ;
       temp = [] ;
       for j = NZ:-1:1
          if (p_z_grid(j) > -sqrt(2).*one_D_H_t)
             temp = [temp; ...
             mean(mean_theta(j,[2:floor(NY/3)-1])) - ...
             mean(mean_theta(j,[floor(NY*2/3)+1:NY-1]))] ;
          end
       end
       temp2 = find(isnan(temp)) ;
       temp(temp2) = [] ;
       if (isempty(temp))
          T_prime_t = [T_prime_t;0] ;
       else
          T_prime_t = [T_prime_t;mean(temp)] ;
       end
       T_prime_t = mean(T_prime_t) ;
   end
   
% Update timeseries matrices.
   max_temp            = [max_temp;max_temp_t] ;
   min_temp            = [min_temp;min_temp_t] ;
   mean_temp           = [mean_temp;mean_temp_t] ;
   one_D_H             = [one_D_H;one_D_H_t] ;
   one_D_temp          = [one_D_temp;one_D_temp_t] ;
   length              = [length; length_t] ;
   dtheta_dy           = [dtheta_dy;dtheta_dy_t] ;
   Nhalf               = [Nhalf;Nhalf_t]  ;
   Shalf               = [Shalf;Shalf_t] ;
   Nhalf_lost          = [Nhalf_lost;Nhalf_lost_t]  ;
   Shalf_lost          = [Shalf_lost;Shalf_lost_t] ;
   T_prime             = [T_prime; T_prime_t] ;

end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Time series analysis complete; calculate diagnostic parameters.              %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Calculate various parameters from timeseries.
% Heat budgets; based on summing the 2 channel halfs, and integrating the whole
% channel. These two should be identical and very small (roundoff error only if
% the model run was using the 'New flux' DiffuType). Expts 1-4 and the first few
% days of 5 use an older algorithm called 'CRAY model', which is not 100% 
% conservative. 
% N.B. +ve sum means the model is GAINing heat.

channel      = Nhalf+Shalf ;
channel_lost = Nhalf_lost + Shalf_lost ;
sum          = Shalf+Nhalf-Shalf_lost-Nhalf_lost   ;
E            = Shalf - Shalf_lost ;

% Get tinst from model fields. 
epsilon = 0.20 ;
delta_E =  - log(cosh(3))*epsilon*Half_flux*x_domain*cooling_width ;

% Fit spline through E(t)/t.
pp = csapi(E./(secs_in_a_day.*times),(secs_in_a_day.*times)) ;
tinst = fnval(pp,delta_E)./secs_in_a_day ;

if( tinst > times(i))
   fprintf(' WARNING !  There are not enough timesteps to reach tinst.\n\n') ;
end

% Code to only consider times up to tinst in the fit below.
cnt = 1 ;
for j = 1:size(dirs,1)
   if(times(j)<tinst)
      cnt = cnt + 1 ;
   end
end

% Truncate each vector to tinst.
times([cnt+1:j])      = [] ;
one_D_H([cnt+1:j])    = [] ;
E([cnt+1:j])          = [] ;
T_prime([cnt+1:j])    = [] ;
Shalf([cnt+1:j])      = [] ;
Shalf_lost([cnt+1:j]) = [] ;
Nhalf_lost([cnt+1:j]) = [] ;
Nhalf([cnt+1:j])      = [] ;
sum([cnt+1:j])        = [] ;
i = cnt ;

% Zone_width definition (somwhat arbitrary).
zone_width = 2*cooling_width ;

% Gamma; heat flux parameterisation. Choose from several types. 
[best_gamma1,flux1] = param_G(E, times, T_prime, one_D_H, tinst, zone_width, const_flag) ;
[best_gamma2,flux2] = param_S(E, times, T_prime, one_D_H, tinst, zone_width, const_flag) ;
[best_gamma3,flux3] = param_NI(E, times, T_prime, one_D_H, tinst, zone_width, const_flag) ;
[best_gamma4,flux4] = param_NII(E, times, T_prime, one_D_H, tinst, zone_width, const_flag) ;

answers = [best_gamma1, best_gamma2, best_gamma3, best_gamma4] ;
return
