% Script to create a figure for the ACCP poster. This is based on the
% BigSky abstract, and consists of 3 figures.
% Figure 1 is an along channel average of temperature and M.
% Figure 2 is an along channel average PV. 
% Figure 3 is an along channel average tracer. 
% 
% Author : tom 25/4/95.

% Setup variables for plot; model constants and grids etc.

clear global
clear
close all

global p_x_grid p_y_grid p_z_grid zvort_x_grid zvort_y_grid zvort_z_grid
global theta zvort x_spacing y_spacing z_spacing this_path NX NY NZ
global x_domain y_domain z_domain f N2 time Half_flux g c rho_bar alpha
global PV_y_grid PV_z_grid p_y_grid u_x_grid u_y_grid

secs_in_a_day = 60*60*24 ;
g             = 9.81 ;
c             = 4e3 ;
rho_bar       = 1000 ;
adjustment = [1.25,1,0.9,1] ;
new_size   = [8.3,6.5] ;
font_size  = 12 ;
font_name  = 'Helvetica' ;
days = [9] ;

Expt1
setup_grids(x_spacing, y_spacing, z_spacing) ;

% Start figure  now.


% Define how each subplot is stretched form the matlab default.
   adjustment = [1.25,1,0.9,1] ;

% Setup grids.
   abs_grid  = p_y_grid./1000 ;
   ord_grid  = p_z_grid(1:NZ) ;
   abs_len   = [0,y_domain]./1000 ;
   ord_len   = [-z_domain,0] ;
   abs_N     = NY ;
   ord_N     = NZ ;
   new_abs_N = NY ;
   new_ord_N = NZ * 2 ;
   Grid_type = 0 ;
   slice_no  = 19 ;

% Surface forcing. See notes in white lab book about definition of flux used (23/1/95);
% it has several peculiar and unexpected features. Comment out peculiarities for plots; 
% they make them look messy, and are minor modifications. 
   flux     = Half_flux.*(tanh((2.*(p_y_grid +125 - 15000))./10000) + 1 ) ;
%flux(1:9) = zeros(size(1:9)) ;
%flux(NY)  = 0 ;
%flux(1)   = 0 ;
   flux     = [p_y_grid',flipud(flux')] ;

   if( std(diff(abs_grid)) > eps | std(diff(ord_grid)) > eps )
      Grid_type = 1 ;
      x = [abs_grid(1) : (abs_grid(abs_N)-abs_grid(1))/(new_abs_N-1) : abs_grid(abs_N)]  ;
      y = [ord_grid(1) : (ord_grid(ord_N)-ord_grid(1))/(new_ord_N-1) : ord_grid(ord_N)]  ;
      [X,Y] = meshgrid(x,y)   ;
      [old_x,old_y] = meshgrid(abs_grid,ord_grid) ;
   else
      [X,Y] = meshgrid(abs_grid,ord_grid) ;
   end

% Setup contouring levels.
   theta_levels = [8:0.05:12] ;
   theta_labels = theta_levels ;
   theta_style    = '-' ;
   ang_mom_levels = [0:-x_domain/20:-x_domain].*f ;
   ang_mom_labels = ang_mom_levels ;
   ang_mom_style  = ':' ;

% Read u and theta field from disk.
   path = [this_path,'/',num2str(days*60*24,5)] ;
   time        = days * secs_in_a_day ;
   
   theta       = readmodel('theta.sun.b',path,'float32') ;
   u           = readmodel('u.sun.b',path,'float32') ;

% Get max, min values and remove land from slices.
   mask  = find(theta < 1) ;
% Get max values
   temp  = -99.*ones(size(mask)) ;
   theta(mask)  = temp ;
   max_theta         = max(max(theta)) ;

% Get min values
   temp  = 99.*ones(size(mask)) ;
   theta(mask)  = temp ;
   min_theta         = min(min(theta))  ;

% Remove land.
   temp  = NaN.*ones(size(mask)) ;
   theta(mask)  = temp ;
   mask  = ~isnan(theta) ;

% Calculate angular momentum.
   fprintf(1,' Calculating angular momentum ... \n') ;
   ang_mom = zeros(NY,NZ*NX) ;
   u       = regrid(u,'x',u_x_grid,p_x_grid,x_domain) ;
   temp    = reshape(u,NY,NX*NZ) ;
   temp2   = meshgrid(f.*u_y_grid,[1:NX*NZ])' ;
   ang_mom = temp - temp2 ;
   ang_mom = reshape(ang_mom,NY*NZ,NX) ;
   clear temp2 u 

% Get means for each field.
   theta_mean     = meanyz(theta,x_spacing,mask) ;
   ang_mom_mean   = meanyz(ang_mom,x_spacing,mask) ;

% Flip fields so the cooled region is to the North.
   theta_mean    = fliplr(theta_mean) ;
   ang_mom_mean  = fliplr(ang_mom_mean) ;

% Regrid if needs be (z-grid is uneven).
   if( Grid_type == 1)
      fprintf(1,' Regridding %d data ... \n',new_abs_N*new_ord_N*2) ;
      theta_mean    = interp2(old_x,old_y,theta_mean,X,Y) ;
      ang_mom_mean  = interp2(old_x,old_y,ang_mom_mean,X,Y) ;
   end

   clear theta ang_mom

% Save plotting of these lines until PV colour map is done (otherwise
% the lines are obscured).

% Setup grids.
   abs_grid  = PV_y_grid./1000 ;
   ord_grid  = PV_z_grid(1:NZ) ;
   abs_len   = [0,y_domain]./1000 ;
   ord_len   = [-z_domain,0] ;
   abs_N     = NY ;
   ord_N     = NZ ;
   new_abs_N = NY ;
   new_ord_N = NZ * 2 ;

   if( std(diff(abs_grid)) > eps | std(diff(ord_grid)) > eps )
      Grid_type = 1 ;
      x = [abs_grid(1) : (abs_grid(abs_N)-abs_grid(1))/(new_abs_N-1) : abs_grid(abs_N)]  ;
      y = [ord_grid(1) : (ord_grid(ord_N)-ord_grid(1))/(new_ord_N-1) : ord_grid(ord_N)]  ;
      [X,Y] = meshgrid(x,y)   ;
      [old_x,old_y] = meshgrid(abs_grid,ord_grid) ;
   else
      [X,Y] = meshgrid(abs_grid,ord_grid) ;
   end

% Setup contouring levels.
   PV_levels = [0.01,0.3,0.6,0.9] ;
   PV_labels = PV_levels ;    
   PV_style    = '-' ;
   
% Read in PV from disk.
   fprintf(1,' Day [%d] ... \n',days) ;
   path = [this_path,'/',num2str(days*60*24,5),'/q.mat'] ;
   fid  = fopen(path,'r') ;
   if ( fid == -1)
      fprintf(1,' ERROR !  Cannot read file [%s]. \n\n', path) ;
      return ;
   end
   q = fread(fid,[NY*NZ,NX],'float32') ;
   fclose(fid) ;
   q = -q.*g/(f*N2) ;
  
% Get max, min values and remove land from slices.
   mask     = find(isnan(q)) ;
   q2       = q ;
% Get max values
   temp     = -99.*ones(size(mask)) ;
   q2(mask) = temp ;
   max_q    = max(max(q2)) ;

% Get min values
   temp     = 99.*ones(size(mask)) ;
   q2(mask) = temp ;
   min_q    = min(min(q2))  ;

% Set mask for average function below.
   mask     = ~isnan(q) ;

% Get mean.
  PV_mean     = meanyz(q,x_spacing,mask) ;

% Flip fields so the cooled region is to the North.
   PV_mean    = fliplr(PV_mean) ;

% Regrid if needs be (z-grid is uneven).
   if( Grid_type == 1)
      fprintf(1,' Regridding %d data ... \n',new_abs_N*new_ord_N) ;
      PV_mean    = interp2(old_x,old_y,PV_mean,X,Y) ;
   end

   clear q q2 mask

% Do tracer now.

% Setup grids.
   abs_grid  = p_y_grid./1000 ;
   ord_grid  = p_z_grid(1:NZ) ;
   abs_len   = [0,y_domain]./1000 ;
   ord_len   = [-z_domain,0] ;
   abs_N     = NY ;
   ord_N     = NZ ;
   new_abs_N = NY ;
   new_ord_N = NZ * 2 ;

   if( std(diff(abs_grid)) > eps | std(diff(ord_grid)) > eps )
      Grid_type = 1 ;
      x = [abs_grid(1) : (abs_grid(abs_N)-abs_grid(1))/(new_abs_N-1) : abs_grid(abs_N)]  ;
      y = [ord_grid(1) : (ord_grid(ord_N)-ord_grid(1))/(new_ord_N-1) : ord_grid(ord_N)]  ;
      [X,Y] = meshgrid(x,y)   ;
      [old_x,old_y] = meshgrid(abs_grid,ord_grid) ;
   else
      [X,Y] = meshgrid(abs_grid,ord_grid) ;
   end

% Read in tracer from disk.     
   fprintf(1,' Day [%d] ... \n',days) ;
   path = [this_path,'/',num2str(days*60*24,5)] ;
   tracer      = readmodel('salt.sun.b',path,'float32') ;

% Get max, min values and remove land from slices.
   mask     = find(tracer<10*eps) ;
   tracer2       = tracer ;
% Get max values
   temp     = -99.*ones(size(mask)) ;
   tracer2(mask) = temp ;
   max_tracer    = max(max(tracer2)) ;

% Get min values
   temp     = 99.*ones(size(mask)) ;
   tracer2(mask) = temp ;
   min_tracer    = min(min(tracer2))  ;

% Set mask for average function below.
   temp  = NaN.*ones(size(mask)) ;
   tracer(mask)  = temp ;
   mask  = ~isnan(tracer) ;

% Normalise tracer values.
   tracer = (tracer - max_tracer + 0.5) .*2 ;

% Get mean.
  tracer_mean     = meanyz(tracer,x_spacing,mask) ;

% Flip fields so the cooled region is to the North.
   tracer_mean    = fliplr(tracer_mean) ;

% Regrid if needs be (z-grid is uneven).
   if( Grid_type == 1)
      fprintf(1,' Regridding %d data ... \n',new_abs_N*new_ord_N) ;
      tracer_mean    = interp2(old_x,old_y,tracer_mean,X,Y) ;
   end


% Start plotting.
   figure

% First plot is theta and angular momentum.
   subplot(3,1,1)
   hold on
   [c2,h] = contour(X,Y,theta_mean,theta_levels,theta_style) ;
   clabel(c2,'manual') ;
   set(h,'Color',[1 1 1]) ;
   [c2,h] = contour(X,Y,ang_mom_mean,ang_mom_levels,ang_mom_style) ;
   set(h,'Color',[1 1 1]) ;
   h = plot(flux(:,1)./1000,mixed_layer(flux(:,2),N2,time),'--') ;
   set(h,'LineWidth',2,'Color',[1 1 1]) ;
   set(gca,'AspectRatio',[1.5,NaN]) ;
   axis([0 y_domain/1000 -2000 0] ) ;

% Second plot is PV.
   subplot(3,1,2) ;
   hold on
   pcolor(X,Y,PV_mean) ;
   shading flat  
   caxis([-0.25, 1.25]) ;
   set(gca,'AspectRatio',[1.5,NaN]) ;
   h = plot(flux(:,1)./1000,mixed_layer(flux(:,2),N2,time),'--') ;
   set(h,'LineWidth',2,'Color',[1 1 1]) ;
   text(35,-2300,'Normalised PV') ;
   axis([0 y_domain/1000 -2000 0] ) ;

% Third plot is tracer.
   subplot(3,1,3) ; 
   hold on
   pcolor(X,Y,tracer_mean) ;
   shading flat
   %caxis([0, 1]) ;
   set(gca,'AspectRatio',[1.5,NaN]) ;
   h = plot(flux(:,1)./1000,mixed_layer(flux(:,2),N2,time),'--') ;
   set(h,'LineWidth',2,'Color',[1 1 1]) ;
   text(37,-2300,'Tracer') ;
   axis([0 y_domain/1000 -2000 0] ) ;

% Annotate subplots.
   subplot(3,1,1)
   text(32,-600, ' -  Temperature (C)') ;
   text(32,-1200,' .. Angular momentum') ;
   text(32,-1800,' -- Simple prediction') ;
   text(-8,15,'a)') ;

   subplot(3,1,2) ;
   ylabel('Depth (m)') ;
   text(-8,15,'b)') ;
   p = get(gca,'Position') ;
   temp = gca ;
   h = colorbar('vert') ;
   set(temp,'Position',p) ;

   subplot(3,1,3) ;
   xlabel('Across channel distance (km).','FontName',font_name,'FontSize',font_size) ;
   text(-8,15,'c)') ;
   p = get(gca,'Position') ;
   temp = gca ;
   h = colorbar('vert') ;
   set(temp,'Position',p) ;

   un = get(gcf,'PaperUnits') ;
   set(gcf,'PaperUnits', 'centimeters') ;
   set(gcf,'PaperPosition',[1.0,2.0,19,25.5]) ;
   set(gcf,'PaperUnits', un) ;

return
