% Script to plot a figure of the heat budget from the CM5 channel model integrations.
% Used for figure 5 of the mixed layer instability paper.
% Author : Tom Haine, Mar '95.

clear
clear global
close all

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Setup variables.                                                             %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

global x_spacing y_spacing z_spacing this_path NX NY NZ
global x_domain y_domain z_domain
global p_x_grid p_y_grid p_z_grid
global secs_in_a_day f g c N2 alpha rho_bar flux time Half_flux Initial_temps

feval('Expt1') ;

% List of dump times to read in and display.
dirs = [ 1440;
         2880;
         4320;
         5760;
         7200;
         8640;
         10080;
         11520;
         12960;
         13440;
         13920;
         14400;
         15360 ] ;
times   = dirs./(24*60)  ;
param_type = 1 ;
param_file_name = 'Expt. 1' ;

% Constants. SI units unless explicitly stated otherwise. NB variables with dimensions of
% time in this script are generally in days. This applies to ; times and tinst.
Earth_rad  = 6370e3 ;
Radian     = 57.29578 ;
g          = 9.81 ;
c          = 4e3 ;
rho_bar    = 1000 ;
True       = 1 ;
False      = 0 ;
secs_in_a_day = 60*60*24 ;
Init_temp  = 12 ;
surface_layer = 19 ;

% Define grids.
setup_grids(x_spacing,y_spacing,z_spacing) ;

% Surface forcing. See notes in white lab book about definition of flux used (23/1/95);
% it has several peculiar and unexpected features.
flux     = Half_flux.*(tanh((2.*(p_y_grid +125 - 15000))./10000) + 1 ) ;
flux(1:9) = zeros(size(1:9)) ;
flux(NY)  = 0 ;
flux(1)   = 0 ;
flux     = [p_y_grid',flux'] ;
%mean(flux([2:NY-1],2))

main_dir = this_path ;

% Begin loop.
for i = 1:size(dirs,1)
   this_path = [main_dir,'/',int2str(dirs(i))] ;
   fprintf(1,' \nRunning on directory %s ... ',this_path) ;

% Try to read in  variables from external file; if no file exists read
% in the full model dumps in order.
   temp_file = [main_dir,'/summary/',int2str(dirs(i))] ;
   fid = fopen(temp_file,'r') ;
   if(fid ~= -1)
%  Read in external dump file directly.
      READ_DUMP = 1 ;
      fprintf(1,' Reading dump file. ') ;
      mean_theta         = fread(fid,[NZ,NY], 'float32') ;
      theta_zero_cross_t = fread(fid,1,       'float32') ;
      v_zero_cross_t     = fread(fid,1,       'float32') ;                    
      length_t           = fread(fid,1,       'float32') ;                   
      max_temp_t         = fread(fid,1,       'float32') ;                   
      min_temp_t         = fread(fid,1,       'float32') ;                   
      mean_temp_t        = fread(fid,1,       'float32') ;                   
      one_D_H_t          = fread(fid,1,       'float32') ;                   
      one_D_temp_t       = fread(fid,1,       'float32') ;                   
      dtheta_dy_t        = fread(fid,1,       'float32') ;                   
      Nhalf_t            = fread(fid,1,       'float32') ;                   
      Shalf_t            = fread(fid,1,       'float32') ;                   
      Nhalf_lost_t       = fread(fid,1,       'float32') ;                   
      Shalf_lost_t       = fread(fid,1,       'float32') ;                   
      fclose(fid) ;
   else
      fprintf(1,' ERROR !  Cannot find file [%s]. \n',temp_file) ;
      return ;
   end

% T_prime; the average difference in temperature across the front. This is not
% saved by the code above; so calculate it from the mean theta section.
T_prime_t              = mean(mean_theta(surface_layer , 2:40)) -  ...
                         mean(mean_theta(surface_layer , 81:119)) ;
   
% Update timeseries matrices.
   one_D_H             = [one_D_H;one_D_H_t] ;
   one_D_temp          = [one_D_temp;one_D_temp_t] ;
   dtheta_dy           = [dtheta_dy;dtheta_dy_t] ;
   Nhalf               = [Nhalf;Nhalf_t] ;
   Nhalf_lost          = [Nhalf_lost;Nhalf_lost_t] ;
   Shalf               = [Shalf;Shalf_t] ;
   Shalf_lost          = [Shalf_lost;Shalf_lost_t] ;
   T_prime             = [T_prime; T_prime_t] ;

end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Time series analysis complete; calculate diagnostic parameters.              %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Calculate various parameters from timeseries.
% Heat budgets; based on summing the 2 channel halfs, and integrating the whole
% channel. These two should be identical and very small (roundoff error only if
% the model run was using the 'New flux' DiffuType). Expts 1-4 and the first few
% days of 5 use an older algorithm called 'CRAY model', which is not 100% 
% conservative. 
% N.B. +ve sum means the model is GAINing heat.

sum          = Shalf+Nhalf-Shalf_lost-Nhalf_lost   ;
E            = Shalf - Shalf_lost ;

% Instability timescale; definition based on the time when the southern half of the
% channel loses as much heat sideways to the north as it does through the surface.
tinst        = interp1((Shalf-sum)./(Shalf_lost-sum),times,2,'spline') ;

if( tinst > times(i))
   fprintf(' WARNING !  There are not enough timesteps to reach tinst.\n\n') ;
end

% Gamma; heat flux parameterisation. Choose from 2 types; Martin's and mine.
if (param_type == 1)
   [best_gamma,flux] = param_1(E, times, T_prime, one_D_H, 0) ;
elseif (param_type == 2)
% Choose two estimates of dtheta_dy. First from the model fields themselves, I.e.
% from definition above. Second, from 1D law (see white lab book, 8/2/95).
   dtheta_dy1  = dtheta_dy ;
   dtheta_dy2  = sqrt((2.*times.*secs_in_a_day.*Half_flux.*N2)./(g*alpha*rho_bar*c))./10000 ;
   dtheta_dy   = dtheta_dy1  ;
   [best_gamma,flux] = param_2(E, times, one_D_H, dtheta_dy, 0) ;
else
   fprintf(1, ' ERROR : Illegal parameterisation type [%d]. \n',param_type) ;
   return ;
end

fit = Shalf_lost-flux ;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Display output.                                                              %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Draw a figure of heat budget.
figure
set(gca, 'FontName', 'Times', 'FontSize', 12) ;
times      = [0;times] ;
Shalf      = [0;Shalf] ;
Shalf_lost = [0;Shalf_lost] ;
sum        = [0;sum] ;
fit        = [0;fit] ;

plot(times, Shalf-sum, '+') ;
hold on
plot(times, Shalf_lost-sum, '*') ;
plot(times, Shalf-Shalf_lost, 'o') ;
plot(times, fit, '-.') ;
plot([tinst,tinst],[min([Shalf;Shalf_lost]),0],':') ;
xlabel('Model daynumber', 'FontSize', 14) ;
ylabel('Heat (J)', 'FontSize', 14) ;
axis([0 times(i) min([Shalf;Shalf_lost]) 0]) ;

%text(1,min([Shalf;Shalf_lost])*0.5,'+ = H; * = Hsurf; o = Heddy', 'FontSize', 14) ;
message = ['Instability time scale = [',num2str(tinst),'] days.'] ;
text(1, min([Shalf;Shalf_lost])*0.7, message, 'FontSize', 14) ;

hold off

return
