% Script to create 3 figures for the Big Sky meeting.
% Figure 1 is an xy slice of temperature at day 6.
% Figure 2 is an along chaneel average temperature / angular momentum 
% at day 9.
% Figure 3 is an along channel average PV plot at day 9.
% 
% Author : tom 10/3/95.

% Setup variables for plot; model constants and grids etc.

clear global
clear
close all

global p_x_grid p_y_grid p_z_grid zvort_x_grid zvort_y_grid zvort_z_grid
global theta zvort x_spacing y_spacing z_spacing this_path NX NY NZ
global x_domain y_domain z_domain f N2 time Half_flux g c rho_bar alpha
global PV_y_grid PV_z_grid p_y_grid u_x_grid u_y_grid

secs_in_a_day = 60*60*24 ;
g             = 9.81 ;
c             = 4e3 ;
rho_bar       = 1000 ;
adjustment = [1.25,1,0.9,1] ;

% Values for BigSky abstract (small picture).
%new_size   = [8.3,6.5] ;
%font_size  = 9;

% Values for half page picture.
new_size   = 2*[8.3,6.5] ;
font_size  = 12;
font_name  = 'Times' ;

Expt1
setup_grids(x_spacing, y_spacing, z_spacing) ;

title1  = [' Big Sky Figure 2. '];
message = [' Hit <y> to see a plot of ',title1] ;
getkey1 = input( message,'s') ;

title2  = [' Big Sky Figure 3. '];
message = [' Hit <y> to see a plot of ',title2] ;
getkey2 = input( message,'s') ;


if(getkey1 == 'y' | getkey1 == 'Y')
   abs_grid  = p_x_grid./1000 ;
   ord_grid  = p_y_grid./1000 ;
   abs_len   = [0,x_domain]./1000 ;
   ord_len   = [0,y_domain]./1000 ;
   abs_N     = NX ;
   ord_N     = NY ;
   new_abs_N = 100 ;
   new_ord_N = 60 ;
   slice_no  = 19 ;

   if( std(diff(abs_grid)) > eps | std(diff(ord_grid)) > eps )
   %if 1
      Grid_type = 1 ;
      x = [abs_grid(1) : (abs_grid(abs_N)-abs_grid(1))/(new_abs_N-1) : abs_grid(abs_N)]  ;
      y = [ord_grid(1) : (ord_grid(ord_N)-ord_grid(1))/(new_ord_N-1) : ord_grid(ord_N)]  ;
      [X,Y] = meshgrid(x,y)   ;
      [old_x,old_y] = meshgrid(abs_grid,ord_grid) ;
   else
      [X,Y] = meshgrid(abs_grid,ord_grid) ;
   end

   figure(1)

   days = [6] ;
% Read in temperature field at day 6 from disk.
   for i = 1:size(days,2)
      path = [this_path,'/',num2str(days(i)*60*24,5)] ;
      theta = readmodel('theta.sun.b',path,'float32') ;
      theta_slice = getxy(theta,slice_no) ;

% Flip slice so the cooled region is to the North.
      theta_slice = flipud(theta_slice) ;
      theta_depth    = num2str(p_z_grid(slice_no)) ;

% Get max, min values and remove land from slices.
      mask  = find(theta_slice < 2 ) ;
% Get max values
      temp  = -99.*ones(size(mask)) ;
      theta_slice(mask)  = temp ;
      max_theta         = max(max(theta_slice)) ;

% Get min values
      temp  = 99.*ones(size(mask)) ;
      theta_slice(mask)  = temp ;
      min_theta         = min(min(theta_slice)) ;

% Remove land.
      temp  = NaN.*ones(size(mask)) ;
      theta_slice(mask)  = temp ;

   end
% Regrid if needs be (z-grid is uneven).
   if( Grid_type == 1)
      fprintf(1,' Regridding %d data ... \n',new_abs_N*new_ord_N) ;
      theta_slice   = interp2(old_x,old_y,theta_slice,X,Y) ;
   end



   if 1
      pcolor(X,Y,theta_slice) ;
      shading flat ;
      caxis([min_theta,max_theta]) ;
      hold on
      h = colorbar('vert') ;
      p = get(h,'Position') ;
      set(h,'Position',p.*[1,1.4,0.6,0.75]) ;
      ch = get(h,'Children') ;
      for j = 1:length(ch)
         child = get(ch(j),'Type') ;
         if(size(child,2) == 4)
            if(child == 'text')
               set(ch(j),'FontName',font_name,'FontSize',font_size) ;
               old_pos = get(ch(j),'Position') ;
               set(ch(j),'Position',old_pos.*[2,1,1]) ;
            end
         end
      end
   else
      c2 = contour(p_x_grid./1000,p_y_grid./1000,theta_slice,6) ;
   end
   sub_title = ['Day ',num2str(days(i))] ;
   text(5,5,sub_title,'FontName',font_name,'FontSize', font_size) ;
   xlabel('Along channel distance (km).', 'FontName',font_name,'FontSize',font_size) ;
   ylabel('Across channel distance (km).', 'FontName',font_name,'FontSize',font_size) ;
   set(gca,'AspectRatio',[x_domain/y_domain,1]) ;

   old_units = get(gcf,'Units') ;
   set(gcf,'PaperUnits','centimeters') ;
   old_pos = get(gcf,'PaperPosition') ;
   set(gcf,'PaperPosition',[old_pos(1),old_pos(2),new_size]) ;
   set(gcf,'Units',old_units) ;
   set(gca,'FontName',font_name) ;
   set(gca,'FontSize',font_size) ;
end

% Start figure 2 now.

if(getkey2 == 'y' | getkey2 == 'Y')

% Define how each subplot is stretched form the matlab default.
   adjustment = [1.25,1,0.9,1] ;

   slice_no = 100 ;

% Setup grids.
   abs_grid  = p_y_grid./1000 ;
   ord_grid  = p_z_grid(1:NZ) ;
   abs_len   = [0,y_domain]./1000 ;
   ord_len   = [-z_domain,0] ;
   abs_N     = NY ;
   ord_N     = NZ ;
   new_abs_N = NY ;
   new_ord_N = NZ * 2 ;
   Grid_type = 0 ;
   slice_no  = 19 ;

% Surface forcing. See notes in white lab book about definition of flux used (23/1/95);
% it has several peculiar and unexpected features. Comment out peculiarities for plots; 
% they make them look messy, and are minor modifications. 
   flux     = Half_flux.*(tanh((2.*(p_y_grid +125 - 15000))./10000) + 1 ) ;
%flux(1:9) = zeros(size(1:9)) ;
%flux(NY)  = 0 ;
%flux(1)   = 0 ;
   flux     = [p_y_grid',flipud(flux')] ;

   if( std(diff(abs_grid)) > eps | std(diff(ord_grid)) > eps )
      Grid_type = 1 ;
      x = [abs_grid(1) : (abs_grid(abs_N)-abs_grid(1))/(new_abs_N-1) : abs_grid(abs_N)]  ;
      y = [ord_grid(1) : (ord_grid(ord_N)-ord_grid(1))/(new_ord_N-1) : ord_grid(ord_N)]  ;
      [X,Y] = meshgrid(x,y)   ;
      [old_x,old_y] = meshgrid(abs_grid,ord_grid) ;
   else
      [X,Y] = meshgrid(abs_grid,ord_grid) ;
   end

% Setup contouring levels.
   theta_levels = [8:0.05:12] ;
   theta_labels = theta_levels ;
   theta_style    = '-' ;
   ang_mom_levels = [0:-x_domain/20:-x_domain].*f ;
   ang_mom_labels = ang_mom_levels ;
   ang_mom_style  = ':' ;

   figure(2)

% Read u and theta field at day 9 from disk.
   days = [9] ;
   for i = 1:size(days,2)
      path = [this_path,'/',num2str(days(i)*60*24,5)] ;
      time        = days(i) * secs_in_a_day ;
      path        = [this_path,'/',num2str(days(i)*60*24,5)] ;
   
      theta       = readmodel('theta.sun.b',path,'float32') ;
      u           = readmodel('u.sun.b',path,'float32') ;

% Get max, min values and remove land from slices.
      mask  = find(theta < 1) ;
% Get max values
      temp  = -99.*ones(size(mask)) ;
      theta(mask)  = temp ;
      max_theta         = max(max(theta)) ;

% Get min values
      temp  = 99.*ones(size(mask)) ;
      theta(mask)  = temp ;
      min_theta         = min(min(theta))  ;

% Remove land.
      temp  = NaN.*ones(size(mask)) ;
      theta(mask)  = temp ;
      mask  = ~isnan(theta) ;

% Calculate angular momentum.
      fprintf(1,' Calculating angular momentum ... \n') ;
      ang_mom = zeros(NY,NZ*NX) ;
      u       = regrid(u,'x',u_x_grid,p_x_grid,x_domain) ;
      temp    = reshape(u,NY,NX*NZ) ;
      temp2   = meshgrid(f.*u_y_grid,[1:NX*NZ])' ;
      ang_mom = temp - temp2 ;
      ang_mom = reshape(ang_mom,NY*NZ,NX) ;
      clear temp2 u ;

% Get means for each field.
     theta_mean     = meanyz(theta,x_spacing,mask) ;
     ang_mom_mean   = meanyz(ang_mom,x_spacing,mask) ;

% Flip fields so the cooled region is to the North.
      theta_mean    = fliplr(theta_mean) ;
      ang_mom_mean  = fliplr(ang_mom_mean) ;

% Regrid if needs be (z-grid is uneven).
      if( Grid_type == 1)
         fprintf(1,' Regridding %d data ... \n',new_abs_N*new_ord_N*2) ;
         theta_mean    = interp2(old_x,old_y,theta_mean,X,Y) ;
         ang_mom_mean  = interp2(old_x,old_y,ang_mom_mean,X,Y) ;
      end

% Save plotting of these lines until PV colour map is done (otherwise
% the lines are obscured).

   end

% Setup contouring levels.
   PV_levels = [0.01,0.3,0.6,0.9] ;
   PV_labels = PV_levels ;    
   PV_style    = '-' ;
   
% Read in PV at day 9 from disk.
   days = [ 9] ;
   for i = 1:size(days,2)
      fprintf(1,' Day [%d] ... \n',days(i)) ;
      time = days(i) * secs_in_a_day ;
      path = [this_path,'/',num2str(days(i)*60*24,5),'/q.mat'] ;
      fid  = fopen(path,'r') ;
      if ( fid == -1)
         fprintf(1,' ERROR !  Cannot read file [%s]. \n\n', path) ;
         return ;
      end
      q = fread(fid,[NY*NZ,NX],'float32') ;
      fclose(fid) ;
      q = -q.*g/(f*N2) ;
   
% Get max, min values and remove land from slices.
      mask     = find(isnan(q)) ;
      q2       = q ;
% Get max values
      temp     = -99.*ones(size(mask)) ;
      q2(mask) = temp ;
      max_q    = max(max(q2)) ;

% Get min values
      temp     = 99.*ones(size(mask)) ;
      q2(mask) = temp ;
      min_q    = min(min(q2))  ;

% Set mask for average function below.
      mask     = ~isnan(q) ;

% Get mean.
     PV_mean     = meanyz(q,x_spacing,mask) ;

% Flip fields so the cooled region is to the North.
      PV_mean    = fliplr(PV_mean) ;

% Regrid if needs be (z-grid is uneven).
      if( Grid_type == 1)
         fprintf(1,' Regridding %d data ... \n',new_abs_N*new_ord_N) ;
         PV_mean    = interp2(old_x,old_y,PV_mean,X,Y) ;
      end

% Start plotting.
      if 1 
         pcolor(X,Y,PV_mean) ;
         shading flat  
         caxis([-0.25, 1.25]) ;
         white = 0 ;
         light = 0.2 ;
         dark  = 0.8 ;
         map   = [white.*ones(30,3) ;
                  light.*ones(50,3) ;
                  white.*ones(60,3) ;
                   dark.*ones(20,3) ;
                   dark.*ones(40,3) ] ;
         %colormap(map) ;
         
         hold on
         [c2,h] = contour(X,Y,theta_mean,theta_levels,theta_style) ;
         set(h,'Color',[1 1 1]) ;
         [c2,h] = contour(X,Y,ang_mom_mean,ang_mom_levels,ang_mom_style) ;
         set(h,'Color',[1 1 1]) ;
         h = plot(flux(:,1)./1000,mixed_layer(flux(:,2),N2,time),'--') ;
         set(h,'LineWidth',2,'Color',[1 1 1]) ;
         h = colorbar('vert') ;
         p = get(h,'Position') ;
         set(h,'Position',p.*[1,1.4,0.66,0.75]) ;
         ch = get(h,'Children') ;
         for j = 1:length(ch)
            child = get(ch(j),'Type') ;
            if(size(child,2) == 4)
               if(child == 'text')
                  set(ch(j),'FontName',font_name,'FontSize',font_size) ;
                  old_pos = get(ch(j),'Position') ;
                  set(ch(j),'Position',old_pos.*[1.5,1,1]) ;
               end
            end
         end

      else
         c2 = contour(X,Y,PV_mean,PV_levels) ;
         clabel(c2,'manual') ;
      end
   
      sub_title = ['Day ',num2str(days(i))] ;
      text(5,-1500,sub_title,'FontName',font_name,'FontSize', font_size) ;
      set(gca,'AspectRatio',[1.5,NaN]) ;
      axis([0 y_domain/1000 -2000 0] ) ;

   end

% Annotate subplots.
   set(gca,'FontName',font_name) ;
   set(gca,'FontSize',font_size) ;
   set(gca,'YTickLabels',[-2000,-1500,-500,0]) ;
   set(gca,'YTick',[-2000,-1500,-500,0]) ;
   xlabel('Across channel distance (km).','FontName',font_name,'FontSize',font_size) ;
   old_units = get(gcf,'Units') ;
   set(gcf,'PaperUnits','centimeters') ;
   old_pos = get(gcf,'PaperPosition') ;
   set(gcf,'PaperPosition',[old_pos(1),old_pos(2),new_size]) ;
   set(gcf,'Units',old_units) ;
   text(-1,-1300,'Depth (m)','FontName',font_name,'FontSize',font_size,'Rotation',90) ;


end

return
