% Script to plot a scatter plot of PV vs. N at a given time in the channel
% model. Also plots histograms of w.
%
% Author : twnh 10/5/95.

clear 
close all

global x_spacing y_spacing z_spacing NX NY NZ
global u_x_grid v_x_grid w_x_grid p_x_grid PV_x_grid RiNo_x_grid xvort_x_grid yvort_x_grid zvort_x_grid
global u_y_grid v_y_grid w_y_grid p_y_grid PV_y_grid RiNo_y_grid xvort_y_grid yvort_y_grid zvort_y_grid
global u_z_grid v_z_grid w_z_grid p_z_grid PV_z_grid RiNo_z_grid xvort_z_grid yvort_z_grid zvort_z_grid
global x_domain y_domain z_domain
global u v w q g theta f rho_bar this_path

% Constants. SI units unless explicitly stated otherwise.
Earth_rad  = 6370e3 ;
Radian     = 57.29578 ;
g          = 9.81 ;
c          = 4e3 ;
rho_bar    = 1000 ;
True       = 1 ;
False      = 0 ;
secs_in_a_day = 60*60*24 ;

% Get day to process.
param_file_name =   ...
   input(' Please enter the name of the m-file with the parameters for this run : ','s') ;

folder = input(' Please enter the number of the subdirectory for this experiment : ','s') ;

feval(param_file_name) ;
this_path   = [this_path,'/',folder] ;

% Define grids.
setup_grids(x_spacing,y_spacing,z_spacing) ;

% Read data using external function readmodel.m.
% Read temperature (units C).
theta = readmodel('theta.sun.b',this_path,'float32') ;

% Get max, min values and remove land from slices.
   mask  = find(theta < 2 ) ;
% Get max values
   temp  = -99.*ones(size(mask)) ;
   theta(mask)  = temp ;
   max_theta    = max(max(theta));

% Get min values
   temp  = 99.*ones(size(mask)) ;
   theta(mask)  = temp ;
   min_theta    = min(min(theta)) ;

% Remove land.
   temp         = NaN.*ones(size(mask)) ;
   theta(mask)  = temp ;


% Calculate rho from theta field and EOS.
fprintf(1,' Getting density from theta ... \n') ;
rho = EOS(theta) .* (abs(theta) > eps)  ;

% Calculate the vertical derivative of rho.
fprintf(1,' Getting N2 ... \n') ;
drho_dz = d_dn(rho,'z',p_z_grid,z_domain) ;
N2     = -(g/rho_bar).*drho_dz ;
N2 = reshape(N2,NX*NY*NZ,1) ;

% Get max, min values and remove land from slices.
   N2_mask  = find(isnan(N2)) ;
% Get max values
   temp  = -99.*ones(size(N2_mask)) ;
   N2(N2_mask)  = temp ;
   max_N2         = max(max(N2));

% Get min values
   temp  = 99.*ones(size(N2_mask)) ;
   N2(N2_mask)  = temp ;
   min_N2         = min(min(N2));

% Remove land.
   temp  = NaN.*ones(size(N2_mask)) ;
   N2(N2_mask)  = temp ;

% Read PV.
temp = [this_path,'/','q.mat'] ;
if( fopen(temp) == -1)
   fprintf(1, ' Cannot find PV on disk. Run diagnostic program to create this. \n') ;
   return
else
   fprintf(1,' Retrieving PV from disk ... \n') ;
   fid = fopen(temp,'r') ;
   q = -fread(fid,[NY*NZ,NX],'float32') ;
end
fclose(fid) ;
%q = q.*g./(f.*N2) ;
q  = reshape(q,NX*NY*NZ,1) ;

% Get max, min values and remove land from slices.
   q_mask  = find(isnan(q)) ;
% Get max values
   temp  = -99.*ones(size(q_mask)) ;
   q(q_mask)  = temp ;
   max_q         = max(max(q)) ;

% Get min values
   temp  = 99.*ones(size(q_mask)) ;
   q(q_mask)  = temp ;
   min_q         = min(min(q))  ;

% Remove land.
   temp  = NaN.*ones(size(q_mask)) ;
   q(q_mask)  = temp ;

% Read in w.
% Read w (units Nm^-2s^-1; divide by -(g*rho_bar) to get m/s, +ve upwards).
w = readmodel('w.sun.b',this_path,'float32') ;
w = w./(-g*rho_bar) ;
% Get horizontal @ vertical slices.
vert_slice = getxz(w,110) ;
horz_slice = getxy(w,19) ;
line      =  vert_slice(19,:) ;
depth      = num2str(w_z_grid(19)) ;

w = reshape(w,NX*NY*NZ,1) ;

% Get max, min values and remove land from slices.
   mask     = find(abs(w)<eps) ;
% Get max values
   temp     = -99.*ones(size(mask)) ;
   w(mask)  = temp ;
   max_w    = max(max(w));

% Get min values
   temp     = 99.*ones(size(mask)) ;
   w(mask)  = temp ;
   min_w    = min(min(w));

% Remove land.
   %temp     = NaN.*ones(size(mask)) ;
   w(mask)  = [] ;

figure
subplot(3,1,1) ;
c = contour(w_x_grid./1000,w_y_grid./1000,horz_slice,6) ;
title_text = [param_file_name, '   ',folder] ;
title([title_text,'  Vertical velocity  field at ',depth,'m depth.'],'FontSize',14) ;
xlabel('Along channel distance (km).') ;
ylabel('Across channel distance (km).') ;

subplot(3,1,2) ;
c = contour(w_x_grid./1000,w_z_grid,vert_slice,5) ;
%clabel(c) ;
%pcolor(p_x_grid./1000,p_y_grid./1000,theta_slice) ;
%shading flat
%caxis([11.6 12]) ;
title([title_text,'  Vertical velocity field in active convection patch.'],'FontSize',14) ;
xlabel('Along channel distance (km).') ;
ylabel('Depth (m).') ;
%colorbar('vert') ;

subplot(3,1,3)
plot(w_x_grid,line) ;
xlabel('Along channel distance (km).') ;
ylabel(['w, ms^-1']) ;
title(['Vertical velocity at ',depth,'m in active convection patch']) ;
orient tall

%return
figure
%subplot(2,2,1)
plot(N2,q,'.') ;
axis([min_N2,max_N2,min_q,max_q]) ;
xlabel('N2, s^-2') ;
ylabel('Ertel PV, m^-1s^-1') ;
title_string = [param_file_name, '   ',folder] ;
text(min_N2,max_q*1.2,title_string,'FontSize',14) ;

hold on

% Line defining planetary PV.
plot([min_q*g/f,max_q*g/f],[min_q,max_q],'w--') ;


% Remove the NaNs from the PV and N2 fields.
q(q_mask) = [] ;
N2(N2_mask) = [] ;

figure
% Plot histograms.
subplot(3,1,1)
title_string = [param_file_name, '   ',folder] ;
text(min_N2,10,title_string,'FontSize',14) ;
[n,x] = hist(N2,50) ;
n = n./(NX*NY*NZ) ;
semilogy(x,n,'+') ;
title('Histogram of N2') ;
xlabel('N2, s^-2') ;
ylabel('Frequency') ;

subplot(3,1,2)
[n,x] = hist(q,50) ;
n = n./(NX*NY*NZ) ;
semilogy(x,n,'+') ;
title('Histogram of q') ;
xlabel('q, m^-1s^-1') ;
ylabel('Frequency') ;

subplot(3,1,3)
[n,x] = hist(w,50) ;
n = n./(NX*NY*NZ) ;
semilogy(x,n,'+') ;
title('Histogram of w') ;
xlabel('w, ms^-1') ;
ylabel('Frequency') ;

orient tall

