% Script to plot a figure of vertical velocities from the hydrostatic and
% non-hydrostaic versions of the model.
%
% Author : twnh 23/6/95.

clear 
clear global
close all

global x_spacing y_spacing z_spacing NX NY NZ
global u_x_grid v_x_grid w_x_grid p_x_grid PV_x_grid RiNo_x_grid xvort_x_grid yvort_x_grid zvort_x_grid
global u_y_grid v_y_grid w_y_grid p_y_grid PV_y_grid RiNo_y_grid xvort_y_grid yvort_y_grid zvort_y_grid
global u_z_grid v_z_grid w_z_grid p_z_grid PV_z_grid RiNo_z_grid xvort_z_grid yvort_z_grid zvort_z_grid
global x_domain y_domain z_domain
global u v w q g theta f rho_bar this_path

% Constants. SI units unless explicitly stated otherwise.
Earth_rad  = 6370e3 ;
Radian     = 57.29578 ;
g          = 9.81 ;
c          = 4e3 ;
rho_bar    = 1000 ;
True       = 1 ;
False      = 0 ;
secs_in_a_day = 60*60*24 ;

% Expts to process. Expt 1 = non-hydro, 13 = hydro, 14 = conv adj.
Expts = [1,14] ;
title1 = 'Non-hydrostatic' ;
%title2 = 'Hydrostatic' ;
title2 = 'Convective Adjustment' ;

% Iteration to process. 
itr_no = '8640' ;

% Slices to show.
y_slice =  107;
z_slice = 13 ;

% Range of w.
lo_w = -0.19;
hi_w = 0.07 ;

% Positions [left bottom width height] for the subplots.
margin = 0.1 ;
key_height = 0.03 ;
sub_plot_ht = (1-key_height-2*margin)/3 ;
sub_plot_width = (1-4*margin)/2 ;
pos = [2*margin                margin+2.2*sub_plot_ht sub_plot_width sub_plot_ht ;
       3*margin+sub_plot_width margin+2.2*sub_plot_ht sub_plot_width sub_plot_ht ;
       2*margin                margin+1.4*sub_plot_ht sub_plot_width sub_plot_ht ;
       3*margin+sub_plot_width margin+1.4*sub_plot_ht sub_plot_width sub_plot_ht ;
       2*margin                margin+0.6*sub_plot_ht sub_plot_width sub_plot_ht*0.5 ;
       3*margin+sub_plot_width margin+0.6*sub_plot_ht sub_plot_width sub_plot_ht*0.5 ;
       0.40                    margin+1.3*sub_plot_ht 0.3            key_height  ] ;

% vector of plot handles.
h = [0 0 0 0 0 0 0] ;

% Tick marks for the plots.
xticks = [0 10 20 30 40 50]  ;
xticks_txt = [' 0'; '10'; '20'; '30'; '40'; '50'] ;
yticks = [0 10 20 30]  ;
yticks_txt = [' 0'; '10'; '20'; '30'] ;
zticks = [-2000, -1500, -1000, -500, 0] ;
zticks_txt = ['2000'; '1500'; '1000'; ' 500'; '   0'] ;
wticks = [-0.04, -0.02, -0.0, 0.02] ;

font_name = ['Times'] ;


% Read in each w field and plot 3 figs.
figure
old_units = get(gcf,'Units') ;
set(gcf,'Units',get(gcf,'PaperUnits')) ;
old = get(gcf,'Position') ;
siz = get(gcf,'PaperSize') ;
set(gcf,'Position',[old(1) old(2) siz(1) siz(2)] ) ;
set(gcf,'Units',old_units) ;

for i =1:2
% Read in w.
% Read w (units Nm^-2s^-1; divide by -(g*rho_bar) to get m/s, +ve upwards).
   feval(['Expt',num2str(Expts(i))]) ;

% Define grids.
setup_grids(x_spacing,y_spacing,z_spacing) ;

   this_path = [this_path,'/',itr_no] ;
   w = readmodel('w.sun.b',this_path,'float32') ;
   w = w./(-g*rho_bar) ;
% Get horizontal @ vertical slices.
   
   vert_slice = getxz(w,y_slice) ;
   horz_slice = flipud(getxy(w,z_slice)) ;
   line      =  vert_slice(z_slice,:) ;
   depth     = num2str(w_z_grid(z_slice)) ;
   cut       = [ w_x_grid(1), w_x_grid(NX) ;
                 w_y_grid(y_slice), w_y_grid(y_slice) ;
                 w_z_grid(z_slice), w_z_grid(z_slice) ] ;

   w = reshape(w,NX*NY*NZ,1) ;

% Get max, min values and remove land from slices. Max and min values seem
% to be on the boundary, and hence not representative of the active convection.
   mask     = find(abs(w)<eps) ;
% Get max values
   temp     = -99.*ones(size(mask)) ;
   w(mask)  = temp ;
   max_w    = max(max(w))  ;

% Get min values
   temp     = 99.*ones(size(mask)) ;
   w(mask)  = temp ;
   min_w    = min(min(w)) ;

% Remove land.
   %temp     = NaN.*ones(size(mask)) ;
   w(mask)  = [] ;

   h(i) = axes('Position',pos(i,:)) ;
   pcolor(w_x_grid./1000,w_y_grid./1000,horz_slice) ;
   caxis([lo_w hi_w] ) ;
   shading flat
   hold on
   plot(cut(1,:)/1000,cut(2,:)/1000,'y--') ;

   set(gca,'FontName',font_name) ;
   xlabel('Along channel distance (km).') ;
   ylabel('Across channel distance (km).') ;
   set(gca,'AspectRatio',[x_domain/y_domain,1]) ;
   set(gca,'XLim',[0 50]) ;
   set(gca,'YLim',[0 30]) ;
   set(gca,'XTick',xticks) ;
   set(gca,'XTickLabels',xticks_txt) ;
   set(gca,'YTick',yticks) ;
   set(gca,'YTickLabels',yticks_txt) ;


   h(i+2) = axes('Position',pos(i+2,:)) ;
   pcolor(w_x_grid./1000,w_z_grid,vert_slice) ;
   caxis([lo_w hi_w] ) ;
   shading flat
   hold on
   plot(cut(1,:)/1000,cut(3,:),'y--') ;
   set(gca,'FontName',font_name) ;
   xlabel('Along channel distance (km).') ;
   ylabel('Depth (m).') ;
   set(gca,'AspectRatio',[2 NaN]) ;
   set(gca,'XLim',[0 50]) ;
   set(gca,'YLim',[-2000 0]) ;
   set(gca,'XTick',xticks) ;
   set(gca,'XTickLabels',xticks_txt) ;
   set(gca,'YTick',zticks) ;
   set(gca,'YTickLabels',zticks_txt) ;


   h(i+4) = axes('Position',pos(i+4,:)) ;
   plot(w_x_grid/1000,line) ;
   set(gca,'FontName',font_name) ;
   xlabel('Along channel distance (km).') ;
   ylabel(['Vertical velocity (m/s).']) ;
   set(gca,'AspectRatio',[2,NaN]) ;
   set(gca,'XLim',[0 50]) ;
   set(gca,'YLim',[lo_w hi_w]) ;
   set(gca,'XTick',xticks) ;
   set(gca,'XTickLabels',xticks_txt) ;
   %set(gca,'YTick',wticks) ;


   orient tall

end

% Annotations, colorbar etc.
h(7) = axes('Position',pos(7,:)) ;
y = [0,1] ;
x = [lo_w:0.001:hi_w] ;
z = [x;x] ;
pcolor(x,y,z) ;
shading flat
caxis([lo_w hi_w]) ;
set(h(7),'YTick',[]) ;
%set(h(7),'XTick',wticks) ;
set(h(7),'FontName',font_name,'FontSize',12) ;
title(['Vertical velocity (m/s).']) ;

axes(h(1)) ;
title(title1,'FontSize',14) ;
axes(h(2)) ;
title(title2,'FontSize',14) ;

wysiwyg
