% Script to print out a figure to show 1D upright convection and verify the
% 1D law. Plots a figure of theta at a specified timestep, a figure of the
% mean temp. and then a timeseries of mixed layer depth and predicted depth.
%
% Author Tom, 1/11/95. 

clear
clear global
close all
whos

global f N2 alpha g c rho_bar 
global x_spacing y_spacing z_spacing this_path NX NY NZ
global u_x_grid v_x_grid w_x_grid p_x_grid PV_x_grid xvort_x_grid
global u_y_grid v_y_grid w_y_grid p_y_grid PV_y_grid xvort_y_grid
global u_z_grid v_z_grid w_z_grid p_z_grid PV_z_grid xvort_z_grid
global x_domain y_domain z_domain
global Half_flux time Initial_temps cooling_width cooling_offset
global u v w

% Plot cross section of this slice
slice = 1 ;

% timestep for displacement vectors (seconds).
step = 1800 ;

% Critical slope for defining mixed layer base.
crit_slope = 5e-5 ;

% Titles for each sub plot.
title1 = '(a) Temperature(-), streamfunction (--) and flow' ;
title2 = '(b)' ;
title3 = '(c)' ;

% Region to zoom in on. (x in km; y in m). reggion is the (approx indices
% and zoom vars control the axis limits.
region = [40:80] ;
zoom_x = [10 20] ;
zoom_y = [-1000 0] ;

% Aspect Ratio. The first number in this expression is the final ratio
% of x to y on the plot.
AsRat = 0.25*1000*(zoom_x(2)-zoom_x(1))/(zoom_y(2)-zoom_y(1)) ;

sub_plot_width = 0.70 ;
sub_plot_ht    = sub_plot_width/AsRat ;
pos = [0.16 0.60 sub_plot_width     sub_plot_ht ;
       0.16 0.30 sub_plot_width/2.5 sub_plot_ht/1.3 ;
       0.63 0.30 sub_plot_width/2.5 sub_plot_ht/1.3 ] ;
h = [0 0 0] ;

% Constants. SI units unless explicitly stated otherwise.
Earth_rad  = 6370e3 ;
Radian     = 57.29578 ;
g          = 9.81 ;
c          = 4e3 ;
rho_bar    = 1000 ; 
True       = 1 ;
False      = 0 ;
secs_in_a_day = 60*60*24 ;

% Read parameter file.
cd /data5/twnh/matlab/2D_channel
param_file_name =   'Expt20' ;
time_series = [param_file_name,'.times'] ;
feval(param_file_name) ;
main_dir = this_path ;

% Define grids.
setup_grids(x_spacing,y_spacing,z_spacing) ;

% Get surface fluxes.
flux     = Half_flux.*(tanh((2.*(p_y_grid - cooling_offset))./cooling_width) + 1 ) ;
flux     = [p_y_grid',flux'] ;

% Local variables.
theta_levels = [ 11:0.02:12]-12 ;
theta_labels = theta_levels ;
theta_labels = 'manual' ;
theta_labels = [] ;
ang_mom_levels = [0:-y_domain/30:-y_domain].*f ;
ang_mom_labels = [] ;
%ang_mom_labels = 'manual' ;
psi_levels = [-2:0.10:2] ;

% Setup definitions for plotting grids.
abs_grid  = p_y_grid./1000 ;
ord_grid  = p_z_grid(1:NZ) ;
abs_len   = [0,y_domain]./1000 ;
ord_len   = [-z_domain,0] ;
abs_N     = NY ;
ord_N     = NZ ;
new_abs_N = NY ;
new_ord_N = NZ * 2 ;
Grid_type = 0 ;

if( std(diff(abs_grid)) > eps | std(diff(ord_grid)) > eps )
   Grid_type = 1 ;
   x = [abs_grid(1) : (abs_grid(abs_N)-abs_grid(1))/(new_abs_N-1) : abs_grid(abs_N)]  ;
   y = [ord_grid(1) : (ord_grid(ord_N)-ord_grid(1))/(new_ord_N-1) : ord_grid(ord_N)]  ;
   [X,Y] = meshgrid(x,y)   ;
   [old_x,old_y] = meshgrid(abs_grid,ord_grid) ;
else
   [X,Y] = meshgrid(abs_grid,ord_grid) ;
end

% Form subset of full grid to plot velocities if needs be.
y_ext = [1:1:NY] ;
z_ext = [1:1:NZ] ;
temp_y_grid = v_y_grid(y_ext)/1000 ;
temp_z_grid = v_z_grid(z_ext) ;

% Scale for grid in plot.
del_y = 0.5*abs(mean(diff(temp_y_grid)));
del_z = 0.5*abs(mean(diff(temp_z_grid)));

% Read in list of parameters for this session. 
folder = input(' Enter the iteration number of the dump to process : ','s') ;
time    = str2num(folder) * 60 ;
this_path   = [this_path,'/',folder] ;

% Flux switched off after 12960 timesteps.
if(folder > 12960)
   time = 12960*60 ;
end

% Read data using external function readmodel.m. 
% Read temperature (units C).
theta = readmodel('theta.sun.b',this_path,'float32') ;

% Read u (units m/s). 
u = readmodel('u.sun.b',this_path,'float32') ;

% Read v (units m/s). 
v = readmodel('v.sun.b',this_path,'float32') ;

% Read w (units Nm^-2s^-1; divide by -(g*rho_bar) to get m/s, +ve upwards). 
w = readmodel('w.sun.b',this_path,'float32') ;
w = w./(-g*rho_bar) ;

% Get streamfunction. 
stream_fn = psi./1e6 ;
[old_xp,old_yp] = meshgrid(xvort_y_grid./1000,xvort_z_grid) ;
stream_fn = interp2(old_xp,old_yp,stream_fn,X,Y) ;

% Rough calculation of angular momentum.
fprintf(1,' Calculating angular momentum ... \n') ;
ang_mom = zeros(NY,NZ*NX) ;
temp  = reshape(u,NY,NX*NZ) ;
temp2 = meshgrid(f.*u_y_grid,[1:NX*NZ])' ;
ang_mom = temp - temp2 ;
ang_mom = reshape(ang_mom,NY*NZ,NX) ;
clear temp ;
clear temp2 ;

% Take slice of data and extract the required data points (a subset of
% the whole set).
new_v_slice = getyz(v,slice) ;
new_v_slice = new_v_slice(z_ext,y_ext);

% Use v mask for the w mean because w=0 both when there is a land cell and a nearby
% boundary.
new_w_slice = getyz(w,slice) ;
new_w_slice = new_w_slice(z_ext,y_ext) ;

% Calculate the distance travelled in a certain time for plot of velocity vectors.
v_max = max(max(new_v_slice))  ;
w_max = max(max(new_w_slice))  ;
new_v_slice = new_v_slice.*step/1000 ;
new_w_slice = new_w_slice.*step ;

[x,y] = meshgrid(temp_y_grid,temp_z_grid)  ;

% Print figures, of theta & ang_mom, flow, PV  and tracer.
figure
un = get(gcf, 'PaperUnits') ;
set(gcf, 'PaperUnits','centimeters') ;
set(gcf, 'PaperPosition', [3, 3,  15,  20]) ;
set(gcf, 'PaperUnits',un) ;

h(1) = axes('Position',pos(1,:)) ;
set(h(1),'ColorOrder',[1 1 1]) ;

plotslice(theta-12,'yz',slice,p_y_grid,p_z_grid,theta_levels,theta_labels,'-') ;
hold on
h = plot(flux(:,1)./1000,mixed_layer(flux(:,2),N2,time),'--') ;
set(h,'LineWidth',2) ;
tom_quiver(x,y,new_v_slice,new_w_slice) ;
contour(X,Y,stream_fn,psi_levels,'--') ;

xlabel('Across channel distance (km)', 'FontSize',12) ;
ylabel('Depth (m)', 'FontSize',12) ;
t = text(zoom_x(1),zoom_y(2)+150,title1, 'FontSize',12) ;
set(t, 'FontSize',12);
set(get(gca,'Title'),'String','') ;
set(gca,'YLim',zoom_y) ;
set(gca,'XLim',zoom_x) ;
set(gca,'AspectRatio',[AsRat,NaN]) ;
set(gca,'FontSize',10) ;
set(gca,'Box','on') ;
set(gca,'xgrid','off') ;
set(gca,'ygrid','off') ;

% Plot 2nd figure of mean theta from this section to show mixed layer base.
temp = getyz(theta,1) ;
mean_profile = mean(temp(:,[2:NY-1])')' ;
h(2) = axes('Position',pos(2,:)) ;
set(h(2),'ColorOrder',[1 1 1]) ;

plot(mean_profile,p_z_grid,'o') ;
hold on
pp = csapi(p_z_grid(2:NZ),mean_profile(2:NZ)) ;
plot(fnval(pp,p_z_grid),p_z_grid,':', 'Linewidth',1) ;
axis([11.5 11.8 -1000 0]) ;
xlabel('Across channel mean', 'FontSize',12) ;
text(11.55,-1250,'temperature (C)', 'FontSize',12) ;
ylabel('Depth (m)', 'FontSize',12) ;
title(title2, 'FontSize',12) ;
set(gca,'FontSize',10) ;
grid on

% Read im timeseries of theta, and get the mean profile. Calculate the mixed layer
% depth then plot versus the 1D prediction.

fid = fopen(time_series) ;
[dirs,cnt] = fscanf(fid,'%g')   ;
fclose(fid) ;
% times is in units of days.
times   = dirs./(24*60)  ;
H_obs = [] ;

for i = 1:size(dirs,1)
   this_path = [main_dir,'/',int2str(dirs(i))] ;
   fprintf(1,' \nRunning on directory %s ... ',this_path) ;

   theta = readmodel('theta.sun.b',this_path,'float32') ;
   temp = getyz(theta,1) ;
   mean_profile = mean(temp(:,[2:NY-1])')' ;
   pp = csapi(p_z_grid(2:NZ),mean_profile(2:NZ)) ;
   z_grid = [-1000:20:0] ;

% Get inverse gradient function : depth as a function of gradient.
   dpp = fnder(pp) ;
   pp = csapi(fnval(dpp,z_grid),z_grid) ;
   H_obs = [H_obs fnval(pp,crit_slope)]  ;
end

h(3) = axes('Position',pos(3,:)) ;
set(h(3),'ColorOrder',[1 1 1]) ;

times_grid = [0:0.02:times(i)] ;
H_theory = mixed_layer(mean(flux(:,2)),N2,times_grid*secs_in_a_day) ;
plot(times_grid,H_theory) ;
hold on
plot(times,H_obs,'o') ;
xlabel('Time (days)', 'FontSize',12) ;
ylabel('Mixed layer depth (m)', 'FontSize',12) ;
title(title3, 'FontSize',12) ;
set(gca,'FontSize',10) ;
axis([0 10 -1000 0]) ;
grid on

fprintf(1,'\n Peak speeds in the (y,z) directions are [%6.5f, %6.5f] m/s.\n', v_max,w_max) ;

wysiwyg
