% Script to plot a figure of the heat budget from the CM5 channel model integrations.
% Used for figure in the mixed layer instability paper.
% Author : Tom Haine, Mar '95.

clear
clear global
close all
more off

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Setup variables.                                                             %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

global x_spacing y_spacing z_spacing this_path NX NY NZ cooling_offset cooling_width
global x_domain y_domain z_domain
global p_x_grid p_y_grid p_z_grid
global secs_in_a_day f g c N2 alpha rho_bar flux time Half_flux Initial_temps

cd /data5/twnh/matlab/channel
feval('Expt1') ;

% Set K constant ?  0 = variable, 1 = constant.
const_flag = 1 ;
zone_width = cooling_width* 2;

% List of dump times to read in and display.
dirs = [ 1440;
         2880;
         4320;
         5760;
         7200;
         8640;
         10080;
         11520;
         12960;
         14400 ] ;
times   = dirs./(24*60)  ;

% Constants. SI units unless explicitly stated otherwise. NB variables with dimensions of
% time in this script are generally in days. This applies to ; times and tinst.
Earth_rad  = 6370e3 ;
Radian     = 57.29578 ;
g          = 9.81 ;
c          = 4e3 ;
rho_bar    = 1000 ;
secs_in_a_day = 60*60*24 ;

% Define grids.
setup_grids(x_spacing,y_spacing,z_spacing) ;
main_dir = this_path ;

% Begin loop.
for i = 1:size(dirs,1)
   this_path = [main_dir,'/',int2str(dirs(i))] ;
   fprintf(1,' \nRunning on directory %s ... ',this_path) ;

% Try to read in  variables from external file; if no file exists read
% in the full model dumps in order.
   temp_file = [main_dir,'/summary/',int2str(dirs(i))] ;
   fid = fopen(temp_file,'r') ;
   if(fid ~= -1)
%  Read in external dump file directly.
      READ_DUMP = 1 ;
      fprintf(1,' Reading dump file. ') ;
      mean_theta         = fread(fid,[NZ,NY], 'float32') ;
      theta_zero_cross_t = fread(fid,1,       'float32') ;
      v_zero_cross_t     = fread(fid,1,       'float32') ;                    
      length_t           = fread(fid,1,       'float32') ;                   
      max_temp_t         = fread(fid,1,       'float32') ;                   
      min_temp_t         = fread(fid,1,       'float32') ;                   
      mean_temp_t        = fread(fid,1,       'float32') ;                   
      one_D_H_t          = fread(fid,1,       'float32') ;                   
      one_D_temp_t       = fread(fid,1,       'float32') ;                   
      dtheta_dy_t        = fread(fid,1,       'float32') ;                   
      Nhalf_t            = fread(fid,1,       'float32') ;                   
      Shalf_t            = fread(fid,1,       'float32') ;                   
      Nhalf_lost_t       = fread(fid,1,       'float32') ;                   
      Shalf_lost_t       = fread(fid,1,       'float32') ;                   
      fclose(fid) ;
   else
      fprintf(1,' ERROR !  Cannot find file [%s]. \n',temp_file) ;
      return ;
   end

% Definition of T_prime; based on the mean difference in temp across
% the front, averaged over the oneD layer depth.
   if(1)
       T_prime_t = [] ;
       temp = [] ;
       for j = NZ:-1:1
          if (p_z_grid(j) > -sqrt(2).*one_D_H_t)
             temp = [temp; ...
             mean(mean_theta(j,[2:floor(NY/3)-1])) - ...
             mean(mean_theta(j,[floor(NY*2/3)+1:NY-1]))] ;
          end
       end
       temp2 = find(isnan(temp)) ;
       temp(temp2) = [] ;
       if (isempty(temp))
          T_prime_t = [T_prime_t;0] ;
       else
          T_prime_t = [T_prime_t;mean(temp)] ;
       end
       T_prime_t = mean(T_prime_t) ;
   end

% Update timeseries matrices.
   one_D_H             = [one_D_H;one_D_H_t] ;
   dtheta_dy           = [dtheta_dy;dtheta_dy_t] ;
   Nhalf               = [Nhalf;Nhalf_t] ;
   Nhalf_lost          = [Nhalf_lost;Nhalf_lost_t] ;
   Shalf               = [Shalf;Shalf_t] ;
   Shalf_lost          = [Shalf_lost;Shalf_lost_t] ;
   T_prime             = [T_prime; T_prime_t] ;

end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Time series analysis complete; calculate diagnostic parameters.              %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% Calculate various parameters from timeseries.
% Heat budgets; based on summing the 2 channel halfs, and integrating the whole
% channel. These two should be identical and very small (roundoff error only if
% the model run was using the 'New flux' DiffuType). Expts 1-4 and the first few
% days of 5 use an older algorithm called 'CRAY model', which is not 100% 
% conservative. 
% N.B. +ve sum means the model is GAINing heat.

channel      = Nhalf+Shalf ;
channel_lost = Nhalf_lost + Shalf_lost ;
sum          = Shalf+Nhalf-Shalf_lost-Nhalf_lost   ;
E            = Shalf - Shalf_lost ;

% Get tinst from model fields.
epsilon = 0.20 ;
delta_E =  - log(cosh(3))*epsilon*Half_flux*x_domain*cooling_width ;

% Fit spline through E(t)/t.
pp = csapi(E./(secs_in_a_day.*times),(secs_in_a_day.*times)) ;
tinst = fnval(pp,delta_E)./secs_in_a_day ;

if( tinst > times(i))
   fprintf(' WARNING !  There are not enough timesteps to reach tinst.\n\n') ;
end

if( tinst > times(i))
   fprintf(' WARNING !  There are not enough timesteps to reach tinst.\n\n') ;
end

% Do parameterised fit.
[best_gamma,flux] = param_NII(E, times, T_prime, one_D_H, tinst, zone_width, const_flag) ;

fit = Shalf_lost-flux ;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Display output.                                                              %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Draw a figure of heat budget.
figure
h = axes('Position',[0.15 0.15 0.77 0.77]) ;
un = get(gcf,'PaperUnits');
set(gcf,'PaperUnits','centimeters') ;
set(gcf,'PaperPosition',[3 10 8.3 8.3]) ;
set(gcf,'PaperUnits',un) ;

times      = [0;times] ;
Shalf      = [0;Shalf] ;
Shalf_lost = [0;Shalf_lost] ;
sum        = [0;sum] ;
fit        = [0;fit] ;
i = i+1 ;

plot(times, sum-Shalf, 'wo',times, sum-Shalf_lost, 'w--',times, sum-fit, 'w-') ;
hold on
xlabel('Time (days)', 'FontSize', 12) ;
ylabel('Heat (J)', 'FontSize', 12) ;
%text(1,14e16,'Heat budget for southern half of channel.','FontSize',10) ;
text_1 = ['Cumulative heat lost'] ;
text_2 = ['Cumulative heat loss from surface'] ;
text_3 = ['Parameterised fit (see text)'] ;
%l = legend(text_1, text_2, text_3,'FontSize',8) ;
%patch([0.5 8.3],[12.2,15.8].*1e16,'k') ;
text(0.5,14.5e16,'  o   Cumulative heat lost','FontSize', 8) ;
text(0.5,13.5e16,'--- Cumulative heat loss from surface','FontSize', 8) ;
text(0.5,12.5e16,'  -   Parameterised fit (see text)','FontSize', 8);

message = ['Instability time scale = [',num2str(tinst),'] days.'] ;
%text(1, -min([Shalf;Shalf_lost])*0.7, message, 'FontSize', 14) ;

% Now repeat for the 2D runs.

clear Nhalf Shalf Nhalf_lost Shalf_lost
cd ../2D_channel
feval('Expt15ii') ;

% List of dump times to read in and display.
dirs = [ 1440;
         2880;
         4320;
         5760;
         7200;
         8640;
         10080;
         11520;
         12960 ] ;
times   = dirs./(24*60)  ;
main_dir = this_path ;

% Begin loop.
for i = 1:size(dirs,1)
   this_path = [main_dir,'/',int2str(dirs(i))] ;
   fprintf(1,' \nRunning on directory %s ... ',this_path) ;

% Try to read in  variables from external file; if no file exists read
% in the full model dumps in order.
   temp_file = [main_dir,'/summary/',int2str(dirs(i))] ;
   fid = fopen(temp_file,'r') ;
   if(fid ~= -1)
%  Read in external dump file directly.
      READ_DUMP = 1 ;
      fprintf(1,' Reading dump file. ') ;
      mean_theta         = fread(fid,[NZ,NY], 'float32') ;
      one_D_H_t          = fread(fid,1,       'float32') ;
      one_D_temp_t       = fread(fid,1,       'float32') ;
      dtheta_dy_t        = fread(fid,1,       'float32') ;
      Nhalf_t            = fread(fid,1,       'float32') ;
      Shalf_t            = fread(fid,1,       'float32') ;
      Nhalf_lost_t       = fread(fid,1,       'float32') ;
      Shalf_lost_t       = fread(fid,1,       'float32') ;
      fclose(fid) ;
   else
      fprintf(1,' ERROR !  Cannot find file [%s]. \n',temp_file) ;
      return ;
   end

% Update timeseries matrices.
   Nhalf               = [Nhalf;Nhalf_t] ;
   Nhalf_lost          = [Nhalf_lost;Nhalf_lost_t] ;
   Shalf               = [Shalf;Shalf_t] ;
   Shalf_lost          = [Shalf_lost;Shalf_lost_t] ;

end

times      = [0;times] ;
Shalf      = [0;Shalf] ;
Shalf_lost = [0;Shalf_lost] ;
i = i+1 ;

%plot(times, Shalf_lost-Shalf, 'x') ;

fprintf(1,'\n\n Parameterised fit is with a gamma of [%6.5f].',best_gamma) ;
if(const_flag == 0)
   fprintf(1,'\n A variable K was used.') ;
else
   fprintf(1,'\n A constant K was used.') ;
end
fprintf(1,'\n\n') ;
grid on

axis([0 10 0 1.6e17]) ;
set(gca,'FontSize',10) ;

wysiwyg
