% Script to read, filter and display tsg data; specifically the gradients in tsg density.
% tsg data has the following properties;
% i) some bad data points (-999.0),
% ii) periods when the ship is moving very slowly,
% iii) interruptions in the data stream (large jumps in distance run variable).

% Data source is from pstar tsg files. Run peos83 to get density (sigt83), then pavrge
% to average the data into 250m bins (can't find another easy way to grid the data),
% Finally, run mlist the distrun and sigma variables to ascii file. Remove header with
% vi, then the file is ready to load directly into matlab.

% Add code to plot a 2nd figure with the equivalent diffusion coefficients histograms
% graphed, as a function of different mixed layer depths.

% This script prints out fancy figures for the mixed layer paper.

% Does fit to STONE parameterisation.

% Author : twnh, apr '95.

tic
close all
clear

cd /data5/twnh/matlab/Instability

bad_data   = -999. ;
spacing    = 3 ;
wave_cut   = 5000 ;
cut_off    = 20 ;

depths = [50,100,200,400] ;

gamma   = 0.4500 ;
g       = 9.81 ;
rho_bar = 1026 ;
f       = 1e-4 ;

margin = 0.1 ;
key_height = 0.03 ;
sub_plot_ht = (1-key_height-2*margin)/3 ;
sub_plot_width = (1-5*margin) ;
pos = [2*margin                margin+2.3*sub_plot_ht  sub_plot_width sub_plot_ht ;
       2*margin                margin+0.5*sub_plot_ht  sub_plot_width sub_plot_ht ] ;
h = [0 0] ;

if(exist('raw_data') == 0)
  file = '/data8/twnh/VIVALDI/tsgVIV.ascii' ;
  raw_data = read(file,'%g %g %g',1,-1,1) ;
end

% Extract raw_x and raw_rho from the data. Convert x to SI from km
raw_x   = raw_data(:,2)*1000 ; ;
raw_rho = raw_data(:,3) ;
raw_no  = size(raw_data,2) ;

% Get mean record separation.
mean_sep = mean(diff(raw_x)) ;
lo_spacing = mean_sep / spacing ;
%hi_spacing = mean_sep * spacing ;
Nyquist = 1/(2*mean_sep) ;

% Check to see if filter will work OK.
if(wave_cut < 1/Nyquist) 
  fprintf(1,' ERROR !  wave_cut [%g] is too low. It should be >= [%g].',wave_cut, 1/Nyquist) ;
  return ;
end

% Process data to remove bad data, and those data with odd spacing.
 
fprintf(1,' Removing bad_data ... ') ;
indices = (raw_rho == bad_data) ;
new_rho = raw_rho(~indices) ;
new_x   = raw_x(~indices) ;
fprintf(1,' removed %g data. \n',sum(indices)) ;

fprintf(1,' Removing outliers ... ') ;
dist    = [0;diff(new_x)] ;
%indices = (dist > hi_spacing | dist < lo_spacing) ;
indices = (dist < lo_spacing) ;
new_rho = new_rho(~indices) ;
new_x   = new_x(~indices) ;
fprintf(1,' removed %g data.\n',sum(indices)) ;

% Generate Butterworth filter based on wave_cut, the cutoff wavelength.
fprintf(1,' Filtering data with low-pass filter of wavelength %g m ... ',wave_cut) ;
mean_sep = mean(diff(new_x)) ;
[b,a] = butter(7,1/(Nyquist*wave_cut)) ;
filtered_rho = filtfilt(b,a,new_rho) ;
fprintf(1,' done.\n') ;

% Do histogram of density gradients. 
fprintf(1,' Calculating normalised histogram of density gradients ... ') ;
grad_rho = abs(gradient(filtered_rho',mean_sep)) ;
[n,x]    = hist(grad_rho,50) ;
% Remove bins with less than cut_off hits in them.
[toti,totj] = find(n < cut_off) ;
n(totj) = [] ;
x(totj) = [] ;
n = n./size(filtered_rho,1) ;
fprintf(1,' done.\n') ;

figure
subplot(3,1,1)
plot(new_x./1000,filtered_rho) ;
xlabel('Distance along track (km)') ;
ylabel('Sea surface density (kgm-3)') ;

subplot(3,1,2)
%h(1) = axes('Position',pos(1,:)) ;
%set(h(1),'ColorOrder',[1 1 1]) ;

semilogy(x,n) ;
hold off
xlabel('Lateral density gradient, kg/m^4') ;
ylabel('Normalised frequency') ;
grid on

% Do series of plots showing the effective diffusion coefficient for a few (constant)
% mixed layer depths.
subplot(3,1,3)
%h(2) = axes('Position',pos(2,:)) ;
%set(h(2),'ColorOrder',[1 1 1]) ;

for i = 1:length(depths)
   K = ((gamma * g^2) / (rho_bar^2 * f^3) ) .* depths(i)^2 .* x.^2 ;
   loglog(K,n) ;
   hold on
   text(K(length(K)),3e-4,num2str(depths(i))) ;
end
loglog([250,250],[1e-4,1],'r') ;
axis([1 1e5 1e-4 1]) ;
ylabel('Normalised frequency') ;
xlabel('Diffusion coefficient (m^2/s)') ;
grid on

orient tall
toc
