% Script to read, filter and display tsg data; specifically the gradients in tsg density.
% tsg data has the following properties;
% i) some bad data points (-999.0),
% ii) periods when the ship is moving very slowly,
% iii) interruptions in the data stream (large jumps in distance run variable).

% Data source is from pstar tsg files. Run peos83 to get density (sigt83), then pavrge
% to average the data into 250m bins (can't find another easy way to grid the data),
% Finally, run mlist the distrun and sigma variables to ascii file. Remove header with
% vi, then the file is ready to load directly into matlab.

% Add code to plot a 2nd figure with the equivalent diffusion coefficients histograms
% graphed, as a function of different mixed layer depths.

% This script prints out fancy figures for the mixed layer paper.

% Does fit for the NEW parameterisation.

% Author : twnh, apr '96.

tic
close all
clear

cd /data5/twnh/matlab/Instability

bad_data   = -999. ;
spacing    = 3 ;
wave_cut   =  5000 ;
cut_off    = 20 ;

depths = [50,100,200,400] ;

gamma   = 0.1000 ;
Lzone   = 20000 ;   
g       = 9.81 ;
rho_bar = 1026 ;
f       = 1e-4 ;

sub_plot_width = 0.70 ;
sub_plot_ht    = 0.23 ;
pos = [0.12  0.7  sub_plot_width sub_plot_ht ;
       0.12  0.4  sub_plot_width sub_plot_ht ;
       0.12  0.1  sub_plot_width sub_plot_ht ] ;
h = [0 0 0] ;

if(exist('raw_data') == 0)
  file = '/data8/twnh/VIVALDI/tsgVIV.ascii' ;
  raw_data = read(file,'%g %g %g',1,-1,1) ;
end

% Extract raw_x and raw_rho from the data. Convert x to SI from km
raw_x     = raw_data(:,2)*1000 ; ;
raw_rho   = raw_data(:,3) ;
raw_no    = size(raw_data,2) ;

% Get mean record separation.
mean_sep = mean(diff(raw_x)) ;
lo_spacing = mean_sep / spacing ;
%hi_spacing = mean_sep * spacing ;
Nyquist = 1/(2*mean_sep) ;

% Check to see if filter will work OK.
if(wave_cut < 1/Nyquist) 
  fprintf(1,' ERROR !  wave_cut [%g] is too low. It should be >= [%g].',wave_cut, 1/Nyquist) ;
  return ;
end

% Process data to remove bad data, and those data with odd spacing.
 
fprintf(1,' Removing bad_data ... ') ;
indices = (raw_rho == bad_data) ;
new_rho = raw_rho(~indices) ;
new_x   = raw_x(~indices) ;
fprintf(1,' removed %g data. \n',sum(indices)) ;

fprintf(1,' Removing outliers ... ') ;
dist    = [0;diff(new_x)] ;
%indices = (dist > hi_spacing | dist < lo_spacing) ;
indices = (dist < lo_spacing) ;
new_rho = new_rho(~indices) ;
new_x   = new_x(~indices) ;
fprintf(1,' removed %g data.\n',sum(indices)) ;

% Generate Butterworth filter based on wave_cut, the cutoff wavelength.
fprintf(1,' Filtering data with low-pass filter of wavelength %g m ... ',wave_cut) ;
mean_sep = mean(diff(new_x)) ;
[b,a] = butter(7,1/(Nyquist*wave_cut)) ;
filtered_rho = filtfilt(b,a,new_rho) ;
fprintf(1,' done.\n') ;

% Do histogram of density gradients. 
fprintf(1,' Calculating normalised histogram of density gradients ... ') ;
grad_rho = abs(gradient(filtered_rho',mean_sep)) ;
[n,x]    = hist(grad_rho,50) ;
% Remove bins with less than cut_off hits in them.
[toti,totj] = find(n < cut_off) ;
n(totj) = [] ;
x(totj) = [] ;
n = n./size(filtered_rho,1) ;
fprintf(1,' done.\n') ;

figure
un = get(gcf, 'PaperUnits') ;
set(gcf, 'PaperUnits','centimeters') ;
set(gcf, 'PaperPosition', [3, 3, 15, 20]) ;   
set(gcf, 'PaperUnits',un) ;

h(1) = axes('Position',pos(1,:) ) ;
plot(new_x./1000,filtered_rho,'w-') ;
xlabel('Distance along track (km)','FontSize',12) ;
ylabel('Sea surface density (kgm    )','FontSize',12) ;
text(-1600,27.40,'-3','FontSize',10,'Rotation',90) ;
set(gca, 'FontSize', 10) ;
text(0,27.6,'(a)','FontSize',12) ;
axis([0 14000 25.5 27.5]) ;
grid on

h(2) = axes('Position',pos(2,:) ) ;
semilogy(x,n,'w-') ;
hold off
xlabel('Lateral density gradient, (kgm    ) ','FontSize',12) ;
text(2.55e-5,3.0e-5,'-4','FontSize',10) ;
ylabel('Normalised frequency','FontSize',12) ;
grid on
set(gca, 'FontSize', 10) ;
axis([0 3.5e-5 1e-4 1]) ;
text(0,2.0,'(b)','FontSize',12) ;

% Do series of plots showing the effective diffusion coefficient for a few (constant)
% mixed layer depths.
h(3) = axes('Position',pos(3,:) ) ;
for i = 1:length(depths)
   K = (gamma * g) / (rho_bar * f) .* depths(i) .* Lzone .* x ;
   loglog(K,n,'w-') ;
   hold on
   text(K(length(K)),3e-4,num2str(depths(i)),'FontSize',10) ;
end
loglog([250,250],[1e-4,1],'w','Linewidth',2.0) ;
ylabel('Normalised frequency', 'FontSize',12) ;
text(1.5e1,1e-5,'Transfer coefficient (m  /s)', 'FontSize',12) ;
text(6.00e2,1.3e-5,'2','FontSize',10) ;
grid on
set(gca, 'FontSize', 10) ;
axis([1e0 1e4 1e-4 1]) ;
text(1.0e0,2.0,'(c)','FontSize',12) ;
%set(gca,'YTick',[1e-4 1e-3 1e-2 1e-1 1e0]) ;
%set(gca,'YTickLabels',[-4 -3 -2 -1 0]) ;

wysiwyg

toc
