% Script to read and display a figure from the mixed layer paper.
% This function plots diagrams of the mean temperature and angular
% momentum at different times, in the model.
% Model run Expt1viii is used for the output.
% 
% Author : tom 16/2/95.

% Setup variables for plot; model constants and grids etc.

clear global
clear
close all

global p_x_grid p_y_grid p_z_grid u_x_grid u_y_grid u_z_grid
global theta zvort x_spacing y_spacing z_spacing this_path NX NY NZ
global x_domain y_domain z_domain f N2 time Half_flux g c rho_bar alpha

secs_in_a_day = 60*60*24 ;
g             = 9.81 ;
c             = 4e3 ;
rho_bar       = 1000 ;
colors = 'gray' ;

cd /data5/twnh/matlab/channel
Expt1

% Setup grids.
setup_grids(x_spacing, y_spacing, z_spacing) ;
abs_grid  = p_y_grid./1000 ;
ord_grid  = p_z_grid(1:NZ) ;
abs_len   = [0,y_domain]./1000 ;
ord_len   = [-z_domain,0] ;
abs_N     = NY ;
ord_N     = NZ ;
new_abs_N = NY ;
new_ord_N = NZ * 2 ;
Grid_type = 0 ;
title1 = ['Zonal average temperature (-) and absolute momentum (:)'] ;
title2 = ['Zonal average potential vorticity'] ;

fprintf(1,'\n Script to plot out zonal means from the channel model.\n') ;
fprintf(1,'\n Options :  (1) Figure of T and M at days 3, 6, 9,') ;
fprintf(1,'\n            (2) Figure of PV at days 3, 6, 9.') ;
fprintf(1,'\n') ;
opt = input(' Enter the figure required : ') ;
if (opt < 1 | opt > 2 | opt ~= round(opt) )
    fprintf(1,' Enter options 1-2 only.\n\n') ;
    return ;
end

zoom_x = [0 30] ;
zoom_y = [-1000 0] ;
AsRat = 0.25*1000*(zoom_x(2)-zoom_x(1))/(zoom_y(2)-zoom_y(1))/3 ;
margin = 0.1 ;
sub_plot_width = 0.7 ; 
sub_plot_ht    = sub_plot_width/AsRat ;
pos = [0.12  margin+2.2*sub_plot_ht  sub_plot_width sub_plot_ht ;
       0.12  margin+1.1*sub_plot_ht  sub_plot_width sub_plot_ht ;
       0.12  margin+0.0*sub_plot_ht  sub_plot_width sub_plot_ht ;
       0.92  margin+1.25*sub_plot_ht 0.03           0.20    ] ;
h = [0 0 0 0] ;
range = [-0.5 1.0] ;

% Surface forcing. See notes in white lab book about definition of flux used (23/1/95);
% it has several peculiar and unexpected features. Comment out peculiarities for plots; 
% they make them look messy, and are minor modifications. 
flux     = Half_flux.*(tanh((2.*(p_y_grid +125 - 15000))./10000) + 1 ) ;
%flux(1:9) = zeros(size(1:9)) ;
%flux(NY)  = 0 ;
%flux(1)   = 0 ;
flux     = [p_y_grid',flux'] ;

if( std(diff(abs_grid)) > eps | std(diff(ord_grid)) > eps )
   Grid_type = 1 ;
   x = [abs_grid(1) : (abs_grid(abs_N)-abs_grid(1))/(new_abs_N-1) : abs_grid(abs_N)]  ;
   y = [ord_grid(1) : (ord_grid(ord_N)-ord_grid(1))/(new_ord_N-1) : ord_grid(ord_N)]  ;
   [X,Y] = meshgrid(x,y)   ;
   [old_x,old_y] = meshgrid(abs_grid,ord_grid) ;
else
   [X,Y] = meshgrid(abs_grid,ord_grid) ;
end

% Setup contouring levels.
theta_levels = [8:0.05:12] ;
theta_labels = theta_levels ;
theta_style    = '-' ;
ang_mom_levels = [0:-x_domain/20:-x_domain].*f ;
ang_mom_labels = ang_mom_levels ;
ang_mom_style  = ':' ;

figure
un = get(gcf, 'PaperUnits') ;
set(gcf, 'PaperUnits','centimeters') ;
set(gcf, 'PaperPosition', [3, 3, 15, 20]) ;
set(gcf, 'PaperUnits',un) ;

% Read in temperature and u field at days 3, 6 and 9 from disk.
days = [3, 6, 9] ;
for i = 1:size(days,2)
   time        = days(i) * secs_in_a_day ;
   path        = [this_path,'/',num2str(days(i)*60*24,5)] ;

if (opt == 1)
   theta       = readmodel('theta.sun.b',path,'float32') ;
   u           = readmodel('u.sun.b',path,'float32') ;

% Get max, min values and remove land from slices.
   mask  = find(theta < 1) ;
% Get max values
   temp  = -99.*ones(size(mask)) ;
   theta(mask)  = temp ;
   max_theta         = max(max(theta)) ;

% Get min values
   temp  = 99.*ones(size(mask)) ;
   theta(mask)  = temp ;
   min_theta         = min(min(theta))  ;

% Remove land.
   temp  = NaN.*ones(size(mask)) ;
   theta(mask)  = temp ;
   mask  = ~isnan(theta) ;

% Calculate angular momentum.
   fprintf(1,' Calculating angular momentum ... \n') ;
   ang_mom = zeros(NY,NZ*NX) ;
   u       = regrid(u,'x',u_x_grid,p_x_grid,x_domain) ;
   temp    = reshape(u,NY,NX*NZ) ;
   temp2   = meshgrid(f.*u_y_grid,[1:NX*NZ])' ;
   ang_mom = temp - temp2 ;
   ang_mom = reshape(ang_mom,NY*NZ,NX) ;
   clear temp2 u ;

% Get means for each field.
  theta_mean     = meanyz(theta,x_spacing,mask) ;
  ang_mom_mean   = meanyz(ang_mom,x_spacing,mask) ;

% Regrid if needs be (z-grid is uneven).
   if( Grid_type == 1)
      fprintf(1,' Regridding %d data ... \n',new_abs_N*new_ord_N*4) ;
      theta_mean    = interp2(old_x,old_y,theta_mean,X,Y) ;
      ang_mom_mean  = interp2(old_x,old_y,ang_mom_mean,X,Y) ;
   end
end

if (opt == 2)

   path = [this_path,'/',num2str(days(i)*60*24,5),'/q.mat'] ;
   fid  = fopen(path,'r') ;
   if ( fid == -1)
      fprintf(1,' ERROR !  Cannot read file [%s]. \n\n', path) ;
      return ;
   end
   q = fread(fid,[NY*NZ,NX],'float32') ;
   fclose(fid) ;
   q = -q.*g/(f*N2) ;

% Get max, min values and remove land from slices.
   mask     = find(isnan(q)) ;
   q2       = q ;
% Get max values
   temp     = -99.*ones(size(mask)) ;
   q2(mask) = temp ;
   max_q    = max(max(q2)) ;

% Get min values
   temp     = 99.*ones(size(mask)) ;
   q2(mask) = temp ;
   min_q    = min(min(q2))  ;

% Set mask for average function below.
   mask     = ~isnan(q) ;

% Get means 
   PV_mean     = meanyz(q,x_spacing,mask) ;

% Regrid if needs be (z-grid is uneven).
   if( Grid_type == 1)
      fprintf(1,' Regridding %d data ... \n',new_abs_N*new_ord_N*2) ;
      PV_mean    = interp2(old_x,old_y,PV_mean,X,Y) ;
   end

end

% Start plotting.
   h(i) = axes('Position',pos(i,:)) ;
   set(h(i),'ColorOrder',[1 1 1]) ;
   if (opt == 1)
      c2 = contour(X,Y,theta_mean,theta_levels,theta_style) ;
      hold on
      c2 = contour(X,Y,ang_mom_mean,ang_mom_levels,ang_mom_style) ;
   end
   if (opt == 2)
       pcolor(X,Y,PV_mean) ;
       shading flat
       colormap(colors)
       caxis([range(1) range(2)]) ;
   end
   hold on
   hh = plot(flux(:,1)./1000,mixed_layer(flux(:,2),N2,time),'--') ;
   set(hh,'LineWidth',2) ;
   hold off
   sub_title = ['Day ',num2str(days(i))] ;
   text(2,-400,sub_title,'FontSize', 12) ;
   set(gca,'AspectRatio',[AsRat,NaN]) ;
   set(gca,'YLim',zoom_y) ;
   set(gca,'XLim',zoom_x) ;
   kids = get(h(i),'Children') ;
   set(kids(3:13),'Linewidth',1) ;
   set(gca,'xgrid','off') ;
   set(gca,'ygrid','off') ;

end

% Annotate subplots.
subplot(h(1)) ;
if (opt == 1)
title(title1,'FontSize',12) ;
else
title(title2,'FontSize',12) ;
end
xlabel('Across channel distance (km)', 'FontSize', 12) ;
ylabel('Depth (m)', 'FontSize', 12) ;
set(gca,'FontSize',10) ;

subplot(h(2)) ;
xlabel('Across channel distance (km)', 'FontSize', 12) ;
ylabel('Depth (m)', 'FontSize', 12) ;
set(gca,'FontSize',10) ;

subplot(h(3)) ;
xlabel('Across channel distance (km)', 'FontSize', 12) ;
ylabel('Depth (m)', 'FontSize', 12) ;
set(gca,'FontSize',10) ;

% Colorbar
if (opt == 2)
h(4) = axes('Position',pos(4,:)) ;
x = [0,1] ;
y = [range(1):0.01:range(2)] ;
z = [y',y'] ;
pcolor(x,y,z) ;
shading flat
set(h(4),'XTick',[]) ;
set(h(4),'Box','on') ;
caxis([range(1) range(2)]) ;

end

wysiwyg
return
