% Script to optimise fit to eddy heat flux as a function of external parameters.
%
% Author twnh Aug '95. 

clear
clear global
close all
more off
t0 = clock ;

% Vectors of parameters for each expt.
global Lx N_th f_coriolis start_index end_index

% Constants.
global rho_bar g c alpha

% Vectors of timeseries for each expt. start_index and end_index indicate which
% part of these vectors refer to which experiment.
global times eddy_E one_D_H dsigma_dy_bar

% Variables to be passed from the setup files.
global f N2 this_path NX NY NZ x_domain Half_flux

% Variables passed to empirical_param objective function to control fitting.
global no_expts no_fits_per_expt no_deg_freedom

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Get input parameters and setup variables.                                    %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Constants. SI units unless explicitly stated otherwise. 
Earth_rad  = 6370e3 ;
Radian     = 57.29578 ;
g          = 9.81 ;
c          = 4e3 ;
rho_bar    = 1000 ;
True       = 1 ;
False      = 0 ;
secs_in_a_day = 60*60*24 ;
secs_in_a_timestep = 60 ;
surface_layer = 19 ;
cooling_width = 10000 ;
no_fits_per_expt = 3  ; 
no_deg_freedom = -1 ;

% Read in list of parameters for this data. This file has each Expt number. 
param_file_name =   ...
   input(' Please enter the name of the data file for this time series analysis : ','s') ;

% Read in the param_file and create matrices of the Expt number.
fid = fopen(param_file_name) ;
if (fid ~= -1)
   [Expts,cnt] = fscanf(fid,'%d ',[1 inf])   ;
   no_expts = cnt ;
   fclose(fid) ;
else
   fprintf(1,'\n\n ERROR !  Cannot open %s.\n\n',param_file_name) ;
   return ;
end

% Setup variables.
Lx           = zeros(no_expts,1) ;
N_th         = zeros(no_expts,1) ;
f_coriolis   = zeros(no_expts,1) ;
start_index  = zeros(no_expts,1) ;
end_index    = zeros(no_expts,1) ;
Half_B       = zeros(no_expts,1) ;

times         = zeros(no_expts*20,1) ;
eddy_E        = zeros(no_expts*20,1) ;
model_E      =  zeros(no_expts*20,1) ;
one_D_H       = zeros(no_expts*20,1) ;
dsigma_dy_bar = zeros(no_expts*20,1) ;
ind           = 1 ;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%      Loop over each model dump reading in data from summary files.                %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

home_dir = feval('cd') ;
root_dir = ['/data23/twnh/'] ;

% Begin loop.
for i = 1:no_expts
   Expt_no = Expts(i) ;
   fprintf(1,' \nReading Expt %d ...', Expt_no) ;

% Evaluate the setup file for this experiment and extract Nth and f.
% Also reads this_path from the setup file.
   setup_cmd = ['Expt',int2str(Expt_no)] ;
   feval(setup_cmd) ;
   f_coriolis(i)     = f ;
   N_th(i)           = sqrt(N2) ;
   Lx(i)             = x_domain ;
   Half_B(i)         = Half_flux*(g*alpha/(rho_bar*c)) ;
   %Bg_star_over_f(i) = cooling_width/(2*sqrt(Half_B(i)*f_coriolis(i))) ;
   Bg_star_over_f(i) = cooling_width/(2*sqrt(Half_B(i))) ;

% Read in times file for this Expt. Note that the times variable is in timesteps.
   times_file = ['Expt',int2str(Expt_no),'.times'] ;
   fid = fopen(times_file) ;
   if (fid ~= -1)
      [dump_times,cnt] = fscanf(fid,'%d')   ;
      no_times = cnt ;
      fclose(fid) ;
   else
      fprintf(1,'\n\n ERROR !  Cannot open %s.\n\n',times_file) ;
      return ;
   end

% Loop over the files in the summary directory and read in each dump.
   for j=1:no_times
      temp_file = [this_path,'/summary/',num2str(dump_times(j),6)] ;
      fprintf(1,' %d',dump_times(j)) ;
      fid = fopen(temp_file,'r') ;
      if(fid ~= -1)
         mean_theta         = fread(fid,[NZ,NY], 'float32') ;
         theta_zero_cross_t = fread(fid,1,       'float32') ;
         v_zero_cross_t     = fread(fid,1,       'float32') ;                    
         length_t           = fread(fid,1,       'float32') ;                   
         max_temp_t         = fread(fid,1,       'float32') ;                   
         min_temp_t         = fread(fid,1,       'float32') ;                   
         mean_temp_t        = fread(fid,1,       'float32') ;                   
         one_D_H_t          = fread(fid,1,       'float32') ;                   
         one_D_temp_t       = fread(fid,1,       'float32') ;                   
         dtheta_dy_t        = fread(fid,1,       'float32') ;                   
         Nhalf_t            = fread(fid,1,       'float32') ;                   
         Shalf_t            = fread(fid,1,       'float32') ;                   
         Nhalf_lost_t       = fread(fid,1,       'float32') ;                   
         Shalf_lost_t       = fread(fid,1,       'float32') ;                   
         fclose(fid) ;
      else
         fprintf(1,'\n\n ERROR !  Cannot find summary file %s.\n\n', temp_file) ;
         return ;
      end

% Extract the eddy heat, 1D mixed layer depth at the channel centre, and form the mean
% density gradient from the temperature field.
      time      = dump_times(j)*secs_in_a_timestep ;
      E         = Shalf_t - Shalf_lost_t ;
      H         = one_D_H_t ;
      T_prime   = mean(mean_theta(surface_layer , 2:40)) -  ...
                  mean(mean_theta(surface_layer , 81:119)) ;
      dtheta_dy = T_prime / cooling_width ;
      dsigma_dy = dtheta_dy*rho_bar*alpha ;

      times(ind)         = time ;
      model_E(ind)       = -0.015*time^1.63*(Half_flux/f)^1.33 ; 
      eddy_E(ind)        = E ;
      one_D_H(ind)       = H ;
      dsigma_dy_bar(ind) = dsigma_dy ;
      ind = ind + 1 ;
   end

% Note down the start and end indices for this expt in the indexing vectors.
   if(i == 1)
      start_index(1) = 1 ;
      start_index(2) = ind ;
   else
      start_index(i+1) = ind  ;
   end
   end_index(i) = ind-1 ;

end

% All data read in successfully. Truncate vectors to correct size.
start_index(i+1)                           = [] ;
times([ind:length(times)])                 = [] ;
eddy_E([ind:length(one_D_H)])              = [] ;
one_D_H([ind:length(one_D_H)])             = [] ;
dsigma_dy_bar([ind:length(dsigma_dy_bar)]) = [] ;

% Calculate mean evolution and std deviation.
temp = eddy_E(end_index) ;
min(temp)
max(temp)
mean(temp)
std(temp)


clear time
% Do plot of eddy energy evolution.
%for i = 1:no_expts
   %subplot(no_expts,1,i)
   %plot(times([start_index(i):end_index(i)]),eddy_E([start_index(i):end_index(i)]),'+') ;
   %axis([0 3.5e6 -3.5e17 0]) ;
   %hold on
%end
%orient tall

figure 
subplot(2,1,1) ;
hold on
for i = 1:no_expts
   plot(times([start_index(i):end_index(i)]),eddy_E([start_index(i):end_index(i)]),'o') ;
   plot(times([start_index(i):end_index(i)]),eddy_E([start_index(i):end_index(i)]),':') ;
   axis([0 3.5e6 -3.5e17 0]) ;
   hold on
end
orient tall
grid
set(gca,'YScale','log');
set(gca,'XScale','log') ;

subplot(2,1,2) ;
hold on
for i = 1:no_expts
   plot(times([start_index(i):end_index(i)]),model_E([start_index(i):end_index(i)]),'o') ;
   plot(times([start_index(i):end_index(i)]),model_E([start_index(i):end_index(i)]),':') ;
   %axis([0 3.5e6 -3.5e17 0]) ;
   hold on
end
orient tall
grid
set(gca,'XScale','log');
set(gca,'YScale','log') ;

figure ;
hold on
for i = 1:no_expts
   plot(times([start_index(i):end_index(i)]),model_E([start_index(i):end_index(i)]),':') ;
   plot(times([start_index(i):end_index(i)]),eddy_E([start_index(i):end_index(i)]),'o') ;
   %axis([0 3.5e6 -3.5e17 0]) ;
   hold on
end
orient tall
%grid
set(gca,'XScale','log');
set(gca,'YScale','log') ;
return


figure ;
hold on
for i = 1:no_expts
   plot(times([start_index(i):end_index(i)]),model_E([start_index(i):end_index(i)])./eddy_E([start_index(i):end_index(i)]),'o') ;
   %axis([0 3.5e6 -3.5e17 0]) ;
   hold on
end
orient tall
grid
set(gca,'XScale','log');
set(gca,'YScale','log') ;



%figure
%for i = 1:no_expts
   %plot(scaled_times([start_index(i):end_index(i)]),eddy_E([start_index(i):end_index(i)]),':') ;
   %axis([0 3.5e6 -3.5e17 0]) ;
   %hold on
%end
%orient tall


% Stuff to fit to the variation in imposed forcing.
%for i = 1:no_expts
   %time = [time;csapi(eddy_E([start_index(i):end_index(i)]),times([start_index(i):end_index(i)]),-2.5e16)] ;
%end
%time

%figure
%forcing = [8;16;4;2;1;4] ;
%subplot(2,1,1)
%plot(1./forcing,time,'+') ;

%subplot(2,1,2)
%plot(log(1./forcing),log(time),'o') ;

% Stuff to fit to the variation in imposed changes in f.
%for i = 1:no_expts
   %time = [time;csapi(eddy_E([start_index(i):end_index(i)]),times([start_index(i):end_index(i)]),-2.5e16)] ;
%end
%time

%figure
%subplot(2,1,1)
%plot(f_coriolis,time,'+') ;

%subplot(2,1,2)
%plot(log(f_coriolis),log(time),'o') ;

pause


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                                   %
%    Do non-linear fit for the accumulated cross-channel eddy heat.                 %
%                                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf(1,'\n\n Beginning optimisation; please wait ...') ;

% Starting guess for alpha coefficients in fit. Green's formula.
% Coefficients are H dsigma_dy Nth
start_gamma = 0.01*1e-10;
start_alpha = [0 2 -1] ;
start_alpha(4) = 2*(start_alpha(1)-start_alpha(2)) - start_alpha(3) - 1 ;
start_alpha(5) = 1-start_alpha(2) ;
start_alpha(6) = 1-start_alpha(1) + start_alpha(2) ;

gamma       = start_gamma ;
alpha_coeff = start_alpha([1:3]) ;
start_score = empirical_param([gamma, alpha_coeff])  ;

% Specify minimum tolerance for variables, objective function and the max no. steps
foptions(1)  = 1 ;
foptions(14) = 1000 ;

% Use fmins to do non-linear optimisation. See empirical_param.m for details of
% objective function minimised.
[best_guess,options] = fmins('empirical_param', [gamma,alpha_coeff], foptions) ;
gamma = best_guess(1) ;
alpha1 = best_guess(2) ;
alpha2 = best_guess(3) ;
alpha3 = best_guess(4) ;
alpha4 = 2*(alpha1-alpha2) - alpha3 - 1 ;
alpha5 = 1-alpha2 ;
alpha6 = 1-alpha1 + alpha2 ;

% Get best guess.
best_score = empirical_param(best_guess) ;

% Display best fit, and goodness of fit.
format short e
clc
fprintf(1,' \n\n') ;
fprintf(1,' Objective function minimised. [start, end] scores = [ %6.3f, %6.3f]. \n',start_score,best_score) ;
if(best_score > 0 & no_deg_freedom > 0)
   fprintf(1,' Probability that this fit is inconsistent         = [ %8.6f]. \n\n', ...
           gammainc(best_score/2,no_deg_freedom/2)) ;
end
fprintf(1,' Parameters in optimised fit. \n') ;
fprintf(1,' gamma       [start,end]               = [ %6.4f, %6.4f], \n', start_gamma, gamma) ;
fprintf(1,' alpha1                                = [ %6.4f, %6.4f], \n', start_alpha(1), alpha1) ;
fprintf(1,' alpha2                                = [ %6.4f, %6.4f], \n', start_alpha(2), alpha2) ;
fprintf(1,' alpha3                                = [ %6.4f, %6.4f], \n', start_alpha(3), alpha3) ;
fprintf(1,' alpha4                                = [ %6.4f, %6.4f], \n', start_alpha(4), alpha4) ;
fprintf(1,' alpha5                                = [ %6.4f, %6.4f], \n', start_alpha(5), alpha5) ;
fprintf(1,' alpha6                                = [ %6.4f, %6.4f], \n', start_alpha(6), alpha6) ;

fprintf(1,'\n\n Eddy density flux ~ \n') ;
fprintf(1,'(%6.4f) rho_bar^(%6.4f) g^(%6.4f) H^(%6.4f) dsigma_dy_bar^(%6.4f) Nth^(%6.4f) f^(%6.4f)\n\n', ...
           gamma, alpha5, alpha6, alpha1, alpha2, alpha3, alpha4) ;

fprintf(1,' Number of iterations [%d], in [%4.1f] minutes. \n\n', options(10), etime(clock,t0)/60) ;
fprintf(1,' There were [%d] independent degrees of freedom.\n\n',no_deg_freedom) ;
