% Function used by diagnostic4.m script to return an objective function
% for a particular parameterisation of eddy flux.
% The parameterisation is ;
% 
% eddy_E = Lx.(c/alpha).gamma.rho_bar^(1-alpha2).g^(1-alpha1+alpha2)
%          N_th^alpha3.f_coriolis^alpha4. integral(one_D_H^(1+alpha1).dsigma_dy_bar^alpha2).dt
%
% We wish to determine gamma, and alpha1-4. See notes in blue lab book too (28/8/95).
%
% Author Tom Aug '95.

function score = empirical_param(vars) ;

% Declare observations from model as global variables.
global Lx N_th f_coriolis times eddy_E one_D_H dsigma_dy_bar start_index end_index

% Make constants global too.
global c g rho_bar alpha

% Variables needed for the spline function
global pp    

% Variables used for housekeeping.
global no_expts no_fits_per_expt no_deg_freedom

% Unpack variables from calling vector.
gamma  = vars(1) ;
alpha1 = vars(2) ;
alpha2 = vars(3) ;
alpha3 = vars(4) ;

% Known parameters.
alpha4 = 2*(alpha1-alpha2) - alpha3 - 1 ;
alpha5 = 1-alpha2 ;
alpha6 = 1-alpha1 + alpha2 ;

% Check that 3 constraints are identically satisfied; each of these should be zero.
resid = [2 + alpha1 - 4*alpha2 - 3*alpha5 + alpha6;1 - alpha3 - alpha4 - 2*alpha6; ...
         alpha2+alpha5 - 1] ;
if(any(resid > 50*eps))
  fprintf(1,' ERROR !  Problem with constraints on the alpha variables. \n') ;
  resid
  return
end

% Loop over each expt and find predicted eddy_E based on proposed parameterisation.
% Form the objective score based on the data/model misfit.

% Form the constant multiplying the integral and the integral's argument.
constant = (c/alpha)*gamma*rho_bar^alpha5*g^alpha6.*Lx.*N_th.^alpha3.*f_coriolis.^alpha4./(1+alpha1) ;
argument = one_D_H.^(1+alpha1).*dsigma_dy_bar.^alpha2 ;

% Model prediction for eddy energy.
model_eddy_E = [] ; 

for i = 1:no_expts    

% Get predicted time series of eddy energy by evaluating integral.
% Use integration of natural cubic splines to the observations of
% one_D_H and dsigma_dy_bar.
   these_times = times([start_index(i):end_index(i)]) ;
   these_args  = argument([start_index(i):end_index(i)]) ;

% Fit cubic spline through timeseries, and get integral in terms of splines.
   pp  = csapi(these_times, these_args) ;
   ipp = fnint(pp) ;

% Loop over times for this expt. Use a subset of the full timeseries to save
% time.
   no_times = length(these_times) ;
   subset_of_these_times = [] ;
   ind = 1 ;
   for j = 1:round(no_times/no_fits_per_expt):no_times       
      subset_of_these_times(ind) = these_times(j) ;
      data_eddy_E  = [data_eddy_E; eddy_E(j)] ;
      ind = ind + 1 ;
   end

% Calculate integrals.
   model_eddy_E =  [model_eddy_E; - constant(i)*fnval(ipp, subset_of_these_times)'] ;
end

% From data/model misfit. May want to normalise the score with a different
% variance later; what should this really be ?
%model_eddy_E
%data_eddy_E
%pause
score = sum(((data_eddy_E - model_eddy_E) ./ mean(eddy_E)).^2) ;

if(no_deg_freedom < 0 )
   no_deg_freedom = length(data_eddy_E) - 5 ;
end

return

