% Function to caluculate the full potential vorticity based on the Boussinesq form of 
% Ertel's PV, from model fields of temperature, Earth's rotation and flow.
% During course of the calculation also save data files of gradient Richardson
% number, non-hydrostatic parameter (delta) and vertical component of vorticity.
% Function returns a matrix of PV values, and 3 grids to define where the PV values
% appear in the model domain.

% NB This function clears the variables theta, u, v and w.

% Use PV = div(rho*total_vorticity)/rhobar.
% This = total_vorticity . div(rho) iff the Earth's rotation vector has no y-component.

% Author twnh, 1/11/94.

function  PV 

global x_spacing y_spacing z_spacing NX NY NZ
global u_x_grid v_x_grid w_x_grid p_x_grid PV_x_grid RiNo_x_grid xvort_x_grid yvort_x_grid zvort_x_grid
global u_y_grid v_y_grid w_y_grid p_y_grid PV_y_grid RiNo_y_grid xvort_y_grid yvort_y_grid zvort_y_grid
global u_z_grid v_z_grid w_z_grid p_z_grid PV_z_grid RiNo_z_grid xvort_z_grid yvort_z_grid zvort_z_grid
global x_domain y_domain z_domain
global u v w q g theta f rho_bar this_path

q = zeros(NY*NZ,NX) ;

fprintf(1,' Calculating potential vorticity ... \n') ;

% Remove land from arrays.
fprintf(1,' Removing land ... \n') ;
mask = find(abs(theta) < eps) ;
temp = NaN.*ones(size(mask)) ;
theta(mask) = temp ;
u(mask)     = temp ;
v(mask)     = temp ;
w(mask)     = temp ;
clear temp mask

% Calculate rho from theta field and EOS.
fprintf(1,' Getting density from theta ... \n') ;
rho = EOS(theta) .* (abs(theta) > eps)  ;
clear global theta 

% calculate density at the vorticity points.
fprintf(1,' Interpolating density values ...\n') ;
rho_at_x_pt = regrid(rho,'z',p_z_grid,xvort_z_grid,z_domain) ;
rho_at_x_pt = regrid(rho,'y',p_y_grid,xvort_y_grid,y_domain) ;

rho_at_y_pt = regrid(rho,'x',p_x_grid,yvort_x_grid,x_domain) ;
rho_at_y_pt = regrid(rho,'z',p_z_grid,yvort_z_grid,z_domain) ;

rho_at_z_pt = regrid(rho,'x',p_x_grid,zvort_x_grid,x_domain) ;
rho_at_z_pt = regrid(rho,'y',p_y_grid,zvort_y_grid,y_domain) ;

% Calculate vertical velocity shears for use in RiNo calc, and vorticity.
% Note that RiNo and delta are both defined at the same gridpoints.

fprintf(1,' Working out RiNo, and delta ...\n') ;
du_dz         = d_dn(u,'z',u_z_grid,z_domain) ;
dv_dz         = d_dn(v,'z',v_z_grid,z_domain) ;
fprintf(1,' Got vertical vorticity \n') ;
du_dz_at_w_pt = regrid(du_dz,'x',yvort_x_grid,w_x_grid,x_domain) ;
dv_dz_at_w_pt = regrid(dv_dz,'y',xvort_y_grid,w_y_grid,y_domain) ;
fprintf(1,' Interpolated vorticities\n') ;
drho_dz       = d_dn(rho,'z',p_z_grid,z_domain) ;
clear rho ;

temp          = (du_dz_at_w_pt.^2 + dv_dz_at_w_pt.^2) ;
clear du_dz_at_w_pt dv_dz_at_w_pt

RiNo = (g/rho_bar).*(drho_dz./temp) ;
delta = f./sqrt(temp) ;

clear drho_dz temp

% Save RiNo for later use.
fid = fopen([this_path,'/RiNo.mat'],'w') ;
fwrite(fid,RiNo,'float32') ;
fclose(fid) ;

% Save delta for later use.
fid = fopen([this_path,'/delta.mat'],'w') ;
fwrite(fid,delta,'float32') ;
fclose(fid) ;

clear RiNo delta

% Calculate the total vorticity at the vorticity points.
fprintf(1,' Calculating total vorticity ...\n') ;
xvort = d_dn(w,'y',w_y_grid,y_domain) -  dv_dz                        ;
clear dv_dz
yvort = du_dz                         -  d_dn(w,'x',w_x_grid,x_domain);
clear du_dz
zvort = d_dn(v,'x',v_x_grid,x_domain) -  d_dn(u,'y',u_y_grid,y_domain) + ...
         ones(NY*NZ,NX).*f ;
clear global u v w 

% Save vorticity for later use.
fid = fopen([this_path,'/xvort.mat'],'w') ;
fwrite(fid,xvort,'float32') ;
fclose(fid) ;
fid = fopen([this_path,'/yvort.mat'],'w') ;
fwrite(fid,yvort,'float32') ;
fclose(fid) ;
fid = fopen([this_path,'/zvort.mat'],'w') ;
fwrite(fid,zvort,'float32') ;
fclose(fid) ;

% calculate rho*vorticity.
fprintf(1,' Calculating density * vorticity ...\n') ;
rho_xvort = rho_at_x_pt .* xvort ;
rho_yvort = rho_at_y_pt .* yvort ;
rho_zvort = rho_at_z_pt .* zvort ;
clear xvort yvort zvort rho_at_x_pt rho_at_y_pt rho_at_z_pt

% take div(rho*vorticity). This is defined at the corners of grid cells.
fprintf(1,' Taking div(density*vorticity) ...\n') ;  
a = d_dn(rho_xvort,'x',xvort_x_grid,x_domain) ; 
q = a ;
fprintf(1,' Done x-component \n') ;
clear rho_xvort 

a = d_dn(rho_yvort,'y',yvort_y_grid,y_domain)  ;   
q = q + a ;
fprintf(1,' Done y-component \n') ;
clear rho_yvort 

a = d_dn(rho_zvort,'z',zvort_z_grid,z_domain)  ;
q = q + a ;
fprintf(1,' Done z-component \n') ;
clear rho_zvort 

% divide by rho_bar.
fprintf(1,' Scaling by mean density ...\n') ;
q = q ./ rho_bar ;

fprintf(1,' All done. \n') ;
return
