% Function to plot an average of data through the model. Can have xy, yz or xz means. 

function slice = plotmean(data,type,spacing,abs_grid,ord_grid,levels,labels,style)     

global NX NY NZ x_domain y_domain z_domain

mask = (abs(data) > eps) ;

% Get the slice from the data using mean primitives.         
if(type == 'xy')
  slice = meanxy(data,spacing,mask) ;
  abs_label = ['Along channel distance (km)'] ;
  ord_label = ['Across channel distance (km)'] ;
  abs_grid  = abs_grid ./ 1000 ;
  ord_grid  = abs_grid ./ 1000 ;
  abs_len   = [0,x_domain]./1000 ;
  ord_len   = [0,y_domain]./1000 ;
  abs_N     = NX ;
  ord_N     = NY ;
  new_abs_N = NX ;
  new_ord_N = NY ;
  title_string = ['Average with depth in channel'] ;
elseif( type == 'yz')
  slice = meanyz(data,spacing,mask) ;
  abs_label = ['Across channel distance (km)'] ;
  ord_label = ['Depth (m)'] ;
  abs_grid  = abs_grid ./ 1000 ;
  abs_len   = [0,y_domain]./1000 ;
  ord_len   = [-z_domain,0] ;
  abs_N     = NY ;
  ord_N     = NZ ;
  new_abs_N = NY ;
  new_ord_N = NZ * 2 ;
  title_string = ['Average along channel'] ;
elseif( type == 'xz')
  slice = meanxz(data,spacing,mask) ;
  abs_label = ['Along channel distance (km)'] ;
  ord_label = ['Depth (m)'] ;
  abs_grid  = abs_grid ./ 1000 ;
  abs_len   = [0,x_domain]./1000 ;
  ord_len   = [-z_domain,0] ;
  abs_N     = NX ;
  ord_N     = NZ ;
  new_abs_N = NX ;
  new_ord_N = NZ * 2 ;
  title_string = ['Average across channel'] ;
else
  fprintf(' ERROR !  Bad type of mean [%s] in plotmean function. \n',type) ;
  return
end

% Only contour non-land values. Most of the land points are already set to NaN by the
% mean function (hence divide by zero warning). There is a chance that there may be 
% a zero remaining in the matrix. Convert this to NaN, and contouring works OK.
% Checking slice == 0 DOESN'T work though.
fprintf(1,' Removing land ... \n') ;
mask = find(abs(slice) < eps) ;
slice(mask) = NaN.*ones(size(mask)) ;

% Check that the abscissa grid and ordinate grid are the correct length.
if(size(slice,1) ~= ord_N)
   fprintf(1,' ERROR !  The ordinate grid is not the right length in plotmean. \n') ;
   return
end
if(size(slice,2) ~= abs_N)
   fprintf(1,' ERROR !  The abscissa grid is not the right length in plotmean.  \n') ;
   return
end

% Do contour plot. Check for uneven grids, and regrid if needs be.
if( std(diff(abs_grid)) > eps | std(diff(ord_grid)) > eps )
   fprintf(1,' Regridding %d data ... \n',new_abs_N*new_ord_N) ;
   x = [abs_grid(1) : (abs_grid(abs_N)-abs_grid(1))/(new_abs_N-1) : abs_grid(abs_N)]  ;
   y = [ord_grid(1) : (ord_grid(ord_N)-ord_grid(1))/(new_ord_N-1) : ord_grid(ord_N)]  ;
   [X,Y] = meshgrid(x,y)   ;
   [old_x,old_y] = meshgrid(abs_grid,ord_grid) ;
   slice = interp2(old_x,old_y,slice,X,Y) ;
else 
   [X,Y] = meshgrid(abs_grid,ord_grid) ;
end

c = contour(X,Y,slice,levels,style) ;
%pcolor(X,Y,slice) ;
%shading flat 
axis([abs_len,ord_len]) ; 
%caxis([-20,0]) ;
%colorbar('vert') ;

% Label.
clabel(c,labels) ;
xlabel(abs_label) ;
ylabel(ord_label) ;
title(title_string) ;
set(get(gca,'Title'), 'FontSize',14 );
set(gca,'YLim',[-2000 0]) ;

return

