% Function to define the C-grids used by the channel model.

% Setup grids. Model uses an Arakawa C-grid, so think of cubes with the p, T, S defined at
% the centre, u at the centre of the x=0 face, v at the centre of the y=0 face, and z at
% centre of the z=0 face. Setup 4 sets of 3 vectors here. Each set has an x-vector, a y-
% vector and a z-vector, showing the explicit location, in m, of the point at which the
% variable is defined.
% The origin in this right-handed coordinate system is the deepest, southernmost,
% western grid cell. I.e. x-increases to the east, y-increases to the north, z-increases
% upwards. Note that all variables have (No_x_pts, No_y_pts, No_z_pts) values in each
% respective direction. To simulate the solid boundary, one layer of these cells has
% to be shaved off with all the p, T, S, u, v, w set == 0. These are the boundary
% conditions. The x-direction, however is periodic, and so there is no lost grid layer in
% this plane (x==0).

% This setup routine works either for evenly spaced _spacing vectors (x and y) or
% unevenly spaced cells. In each case there are the same number of p, T and S points
% in each direction as No_x_pts, No_y_pts and No_z_pts. For the velocity points there
% is one extra value. The model is periodic in each direction, but one layer at the
% bottom of the ocean, and one side has u=v=w=p=T=S == 0, to represent land.

function setup_grids(x_spacing,y_spacing,z_spacing)

global NX NY NZ
global u_x_grid v_x_grid w_x_grid p_x_grid PV_x_grid RiNo_x_grid xvort_x_grid yvort_x_grid zvort_x_grid
global u_y_grid v_y_grid w_y_grid p_y_grid PV_y_grid RiNo_y_grid xvort_y_grid yvort_y_grid zvort_y_grid
global u_z_grid v_z_grid w_z_grid p_z_grid PV_z_grid RiNo_z_grid xvort_z_grid yvort_z_grid zvort_z_grid
global x_domain y_domain z_domain


fprintf(1,' Defining grids. \n') ;
% x grids.
u_x_grid     = [cumsum(x_spacing)] ;
u_x_grid     = u_x_grid - u_x_grid(1) ;
temp         = [u_x_grid,x_domain] ;
temp2        = [0,u_x_grid] ;
p_x_grid     = 0.5*(temp + temp2) ;
p_x_grid     = p_x_grid(2:NX+1) ;
w_x_grid     = p_x_grid ;
v_x_grid     = p_x_grid ;
PV_x_grid    = u_x_grid ;
RiNo_x_grid  = w_x_grid ;
xvort_x_grid = p_x_grid ;
yvort_x_grid = u_x_grid ;
zvort_x_grid = u_x_grid ;

% y grids.
v_y_grid     = [cumsum(y_spacing)] ;
v_y_grid     = v_y_grid - v_y_grid(1) ;
temp         = [v_y_grid,y_domain] ;
temp2        = [0,v_y_grid] ;
p_y_grid     = 0.5*(temp + temp2) ;
p_y_grid     = p_y_grid(2:NY+1) ;
w_y_grid     = p_y_grid ;
u_y_grid     = p_y_grid ;
PV_y_grid    = v_y_grid ;
RiNo_y_grid  = w_y_grid ;
xvort_y_grid = v_y_grid ;
yvort_y_grid = p_y_grid ;
zvort_y_grid = v_y_grid ;


% z grids.
w_z_grid     = -fliplr(cumsum([fliplr(z_spacing)])) ;
temp         = [w_z_grid,0] ;
temp2        = [0,w_z_grid] ;
p_z_grid     = 0.5*(temp + temp2) ;
p_z_grid     = p_z_grid(2:NZ+1) ;
u_z_grid     = p_z_grid ;
v_z_grid     = p_z_grid ;
PV_z_grid    = w_z_grid ; 
RiNo_z_grid  = w_z_grid ;
xvort_z_grid = w_z_grid ;
yvort_z_grid = w_z_grid ;
zvort_z_grid = p_z_grid ;



return
