% Script to display and manipulate SeaSoar files from VIVALDI data.
% See 2nd blue book too (13/11/95).
%
% Author : Tom, Nov '95.

clear
close all

% File definitions.
file = ['MLIST.ASCII'] ;
format = ['%g %g %g %g %g %g'] ;
NX = 75 ;
NY = 65 ;
bad_data = -999 ;

% Positions of plots.
margin = 0.08 ;
key_height = 0.03 ;
sub_plot_ht = (1-key_height-6*margin) ;
sub_plot_width = (1-3*margin) ;
pos = [1.5*margin 4*margin sub_plot_width sub_plot_ht ;
       0.35       1.5*margin 0.3            key_height  ] ;

% vector of plot handles.
h = [0 0] ;

% Range of colourbar.
range  = [26.9 27.5] ;
colors = flipud(gray(65)) ;
colors = colors([2:65],:) ;
%colors = flipud(flipud(jet)) ;
lat_range = [45 47.5] ;

% Constants.
rho_bar = 1026 ;
g = 9.81 ;
c = 4e3 ;
f = 1e-4 ;
secs_in_a_day = 24*60*60 ;
deg_to_meters = 6400e3*2*pi/360 ;

% Mixed layer depths. See notes on 27/10/95 in blue book.
ml_depth_obs = NaN.*ones(NX,1) ;
delta_sigma = 0.05 ;

% Read in data from ascii dump created by mlist.
if(exist('raw_data') == 0)
% Open file.
   handle = fopen(file, 'r');
   if( handle == -1)
     fprintf(1, 'Error opening file %s', file) ;
     fprintf(1, ' \n\n') ;
     stop
   end
   fprintf(1, 'Opened file %s \n', file) ;
   
% Read file & close.
   fprintf(1, 'Reading data ...') ;
   [raw_data,count] = fscanf(handle,format,[5 inf] ) ;
   fprintf(1, ' Read %d records \n', count) ;
   fclose(handle) ;
   fprintf(1, 'Closed file %s \n\n', file) ;
   raw_data = raw_data' ;

end

distance = raw_data(:,2) ;
latitude = raw_data(:,3) ;
pressure = raw_data(:,4) ;
sigma    = raw_data(:,5) ;
buoyancy = -sigma.*g./rho_bar ;

% Reshape matrices.
distance =   fliplr(reshape(distance,NY,NX)) ;
latitude =   fliplr(reshape(latitude,NY,NX)) ;
pressure = - fliplr(reshape(pressure,NY,NX)) ;
sigma    =   fliplr(reshape(sigma   ,NY,NX)) ;
buoyancy =   fliplr(reshape(buoyancy,NY,NX)) ;

% Remove outliers.
mask = find(sigma == -999) ;
sigma(mask) = ones(size(mask)).*NaN ;
buoyancy(mask) = ones(size(mask)).*NaN ;

% Calculate the vertical integral of the buoyancy field using splines.
% Calculate the 1D mixed layer depth at different times.
fprintf(1,' Calculating spline fits and integrals ...') ;

% Butterworth filter for smoothing N2 profile.
order = 7 ;
[b,a] = butter(order,1/4) ;

% surf_sigma for ml depth diagnosis.
temp = sigma(1,:) ;
temp2 = latitude(1,:) ;
mask = find(isnan(temp)) ;
temp(mask) = [] ;
temp2(mask) = [] ;
pp_surf_sigma = csapi(temp2, temp) ;

N2 = NaN.*ones(NY,NX) ;
for i = 1:NX

% Need to remove outliers from the splines. 
   temp_p = pressure(:,i) ;
   temp_b = buoyancy(:,i) ;

   mask = find(isnan(temp_b)) ;
   temp_b(mask) = [] ; 
   temp_p(mask) = [] ; 

   pp = csapi(temp_p,temp_b) ;
   ipp = fnint(pp) ;

   temp_p = pressure(:,i) ;

   if(size(temp_p,1)>3*order)
      dpp = fnder(pp) ;
      N2_temp = fnval(dpp,temp_p)' ;
      filtered_N2_temp = filtfilt(b,a,N2_temp) ;
      index = find(pressure(:,i) == temp_p(1) ) ;
      N2([index:index-1+size(temp_p,1)],i) = filtered_N2_temp' ;  
   end

% Get mixed layer depth based on delta sigma criterion.
   surf_sigma = fnval(pp_surf_sigma,latitude(1,i)) ;
   if(~isnan(surf_sigma))
      flag = 0 ;
      for j = 1:NY
        if(sigma(j,i) >= surf_sigma + delta_sigma & flag == 0)
           ml_depth_obs(i) = pressure(j) ;
           flag = 1 ;
        end
      end
    end

end

% Contour data.
un = get(gcf, 'PaperUnits') ;
set(gcf, 'PaperUnits','centimeters') ;
set(gcf, 'PaperPosition', [2, 10, 18, 12]) ;
set(gcf, 'PaperUnits',un) ;

h(1) = axes('Position',pos(1,:)) ;
set(h(1),'ColorOrder',[1 1 1]) ;
pcolor(latitude,pressure,sigma) ;
hold on
set(h(1),'ColorOrder',[1 1 1]) ;
contour(latitude,pressure,N2,[ 16e-8  16e-8],'w-') ;
xlabel('Latitude (o)', 'FontSize', 12) ;
ylabel('Pressure (db)', 'FontSize', 12) ;
shading flat
set(h(1),'FontSize',10) ;

% Mixed layer depth.
hold on
hh = plot(latitude(1,:), ml_depth_obs,'w--') ;
set(hh,'LineWidth',2) ;

axis([lat_range(1) lat_range(2) -500 0]) ;

% Colorbar
h(2) = axes('Position',pos(2,:)) ;
y = [0,1] ;
x = [range(1):0.01:range(2)] ;
z = [x;x] ;
pcolor(x,y,z) ;
colormap(colors) ;
shading flat
set(h(2),'YTick',[]) ;
title('Potential density (kgm    )', 'FontSize', 12) ;
text(27.40,2.6,'-3', 'FontSize', 9) ;
set(h(2),'Box','on') ;
set(h(2),'FontSize',10) ;

wysiwyg
fprintf(1,'\n\n') ;
